"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConditionManager = void 0;
const Condition_1 = require("../Condition");
const ConditionManagerTypes_1 = require("./ConditionManagerTypes");
const ConditionGroup_1 = require("../ConditionGroup");
const ConstOperand_1 = require("../ConditionOperands/ConstOperand");
const VariableOperand_1 = require("../ConditionOperands/VariableOperand");
const EventParameterOperand_1 = require("../ConditionOperands/EventParameterOperand");
class ConditionManager {
    constructor(instanceApiRef) {
        this.instanceApiRef = instanceApiRef;
        this.conditionGroup = {};
        this.moduleDescriptors = {};
        this.instanceApiRef = instanceApiRef;
    }
    loadConditions() {
        var _a;
        const moduleDescriptor = this.instanceApiRef.getModuleData(this.instanceApiRef.instanceDir);
        for (const [, device] of Object.entries(moduleDescriptor.Devices)) {
            (_a = device.rules) === null || _a === void 0 ? void 0 : _a.forEach((rule, index) => {
                //Create Any Condition Group
                const conditionGroupAny = this.fillConditionGroupWithConditions(ConditionGroup_1.OperatorType.Or, rule.conditions.any, rule, device);
                //Create Every ConditionGroup
                const conditionGroupEvery = this.fillConditionGroupWithConditions(ConditionGroup_1.OperatorType.And, rule.conditions.every, rule, device);
                //create new condition group and add any and every condition groups to it
                const conditionGroupFinal = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                conditionGroupFinal.addCondition(conditionGroupAny);
                conditionGroupFinal.addCondition(conditionGroupEvery);
                this.conditionGroup["rule" + index] = conditionGroupFinal;
            });
        }
    }
    evaluateCondition(id, ...eventCallbackParameters) {
        if (this.conditionGroup[id]) {
            return this.conditionGroup[id].evaluate(eventCallbackParameters);
        }
        else {
            throw Error(`Rule with ${id} ID does not have condition`);
        }
    }
    fillConditionGroupWithConditions(conditionGroupOperatorType, conditionArray, rule, device) {
        const conditionGroup = new ConditionGroup_1.ConditionGroup(conditionGroupOperatorType);
        conditionArray.forEach((condition) => {
            //create Left side
            const { leftOperand, leftOperandValueType } = this.createLeftOperand(condition.left, rule, device);
            //create Right side
            const rightOperand = this.createRightOperand(condition.right, leftOperandValueType);
            conditionGroup.addCondition(new Condition_1.Condition(leftOperand, condition.operator, rightOperand));
        });
        return conditionGroup;
    }
    createLeftOperand(leftCondition, rule, device) {
        switch (leftCondition.type) {
            case ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter: {
                return {
                    leftOperand: new EventParameterOperand_1.EventParameterOperand(leftCondition.name, leftCondition.index),
                    leftOperandValueType: this.getEventCallbackParameterType(this.getRemoteDevice(rule.event, device), rule.event.event, leftCondition.name)
                };
            }
            case ConditionManagerTypes_1.ConditionLeftType.StatusVariable: {
                return {
                    leftOperand: new VariableOperand_1.VariableOperand(this.instanceApiRef, this.instanceApiRef.instanceId, leftCondition.name),
                    leftOperandValueType: this.getVariableType(this.getRemoteDevice(leftCondition, device), leftCondition.name)
                };
            }
        }
    }
    createRightOperand(rightCondition, leftOperandValueType) {
        //On the right side, only ConstOperand is supported for now
        switch (rightCondition.type) {
            case ConditionManagerTypes_1.ConditionRightType.EventCallbackParameter:
                return new EventParameterOperand_1.EventParameterOperand(rightCondition.name, rightCondition.index);
            case ConditionManagerTypes_1.ConditionRightType.StatusVariable:
                return new VariableOperand_1.VariableOperand(this.instanceApiRef, this.instanceApiRef.instanceId, rightCondition.name);
            case ConditionManagerTypes_1.ConditionRightType.Constant:
                //leftOperandValueType 'label' | 'integer' | 'float' | 'string' | 'boolean' | 'json' | 'list' | 'file'
                return new ConstOperand_1.ConstOperand(this.convertType(rightCondition.value, leftOperandValueType));
        }
    }
    getRemoteDevice(leftConditionOrModuleRuleEvent, device) {
        // check that the variable is in other module or not
        if (leftConditionOrModuleRuleEvent.instanceId && leftConditionOrModuleRuleEvent.instanceId != '') {
            const remoteInstanceDir = this.instanceApiRef.instanceDir.replace(this.instanceApiRef.instanceId, leftConditionOrModuleRuleEvent.instanceId);
            if (!this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId]) {
                this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId] = this.instanceApiRef.getModuleData(remoteInstanceDir);
            }
            return this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId]
                .Devices[Object.keys(this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId].Devices)[0]];
        }
        else {
            return device;
        }
    }
    convertType(value, type) {
        switch (type) {
            case 'integer':
            case 'float':
            case 'number': {
                const num = Number(value);
                if (num) {
                    return num;
                }
                else {
                    throw new Error(`Value type is ${type} but the value is not ${type}.`);
                }
            }
            case 'string':
                return value.toString();
            case 'boolean':
                if (value === true || value == 'true' || value === false || value == 'false') {
                    return (value === true || value == 'true');
                }
                else {
                    throw new Error(`Value type is boolean but the value is not boolean.`);
                }
            case 'json':
                if (typeof value === 'object') {
                    JSON.stringify(value);
                    return value;
                }
                else {
                    return JSON.parse(value);
                }
            default:
                return value;
        }
    }
    getVariableType(item, variableName) {
        const valueType = this.getVariableValueType(item.children, variableName);
        return valueType == '' ? this.getVariableValueType(item.extensions, variableName) : valueType;
    }
    getVariableValueType(childrenOrExtensions, variableName) {
        if (childrenOrExtensions) {
            for (const [key, value] of Object.entries(childrenOrExtensions)) {
                if (key == variableName && value.type == 'variable') {
                    return value.valueType;
                }
            }
        }
        return '';
    }
    getEventCallbackParameterType(item, eventName, parameterName) {
        const paramType = this.getEventCallbackParameterValueType(item.children, eventName, parameterName);
        return paramType == '' ? this.getEventCallbackParameterValueType(item.extensions, eventName, parameterName) : paramType;
    }
    getEventCallbackParameterValueType(childrenOrExtensions, eventName, parameterName) {
        if (childrenOrExtensions) {
            for (const [eventKey, eventValue] of Object.entries(childrenOrExtensions)) {
                if (eventKey == eventName && eventValue.type == 'event') {
                    for (const parameter of eventValue.callbackParameters) {
                        if (parameter.name == parameterName) {
                            return parameter.type;
                        }
                    }
                }
            }
        }
        return '';
    }
}
exports.ConditionManager = ConditionManager;
//# sourceMappingURL=ConditionManager.js.map