import { ConditionInterface } from "./ConditionInterface";
import { ConditionOperand, ConditionOperandType } from "./ConditionOperands/ConditionOperand";
import { ConstOperand } from "./ConditionOperands/ConstOperand";
import { EventParameterOperand } from "./ConditionOperands/EventParameterOperand";
import { VariableOperand } from "./ConditionOperands/VariableOperand";

export enum Operator {
  Equals = 'equals',
  NotEquals = 'not_equals',
  Less = 'less',
  LessOrEqual = 'less_or_equal',
  Greater = 'greater',
  GreaterOrEqual = 'greater_or_equal',
  Contains = 'contains',
  Regexp = 'regexp',
  '==' = Equals,
  '!=' = NotEquals,
  '<' = Less,
  '<=' = LessOrEqual,
  '>' = Greater,
  '>=' = GreaterOrEqual,
  'contains' = Contains,
  'regexp' = Regexp
}

/* eslint-disable */
export namespace Operator {
  export function isValidKey(value: string): boolean {
    return (Object.keys(Operator) as string[]).includes(value);
  }

  export function isValidValue(value: string): boolean {
    return (Object.values(Operator) as string[]).includes(value);
  }
}
/* eslint-enable */

export function createCondition(operandLhs: ConditionOperand,
                                operator: Operator | string,
                                operandRhs: ConditionOperand | number | string | boolean,
                                duration: number = 0): Condition {

  if (typeof operandRhs === 'number' || typeof operandRhs === 'string' || typeof operandRhs === 'boolean') {
    const value = operandRhs;
    operandRhs = new ConstOperand(value);
  }

  return new Condition(operandLhs, operator, operandRhs, duration);
}

export class Condition implements ConditionInterface {
  constructor(private readonly operandLhs: ConditionOperand,
              private readonly operator: Operator | string,
              private readonly operandRhs: ConditionOperand,
              private readonly duration: number = 0) {

    if (!Operator.isValidValue(operator)) {
      if (Operator.isValidKey(operator)) {
        this.operator = Operator[operator as keyof typeof Operator] as Operator;
      } else {
        throw Error(`Invalid conditional operator of ${this.operator}`);
      }
    }

    if (this.operator == Operator.Regexp && !(operandRhs instanceof ConstOperand)) {
      throw Error("Right Handside operand must be 'ConstOperand', if the operator is 'Regexp'");
    }

    if (!(this.operandLhs instanceof VariableOperand) && this.duration > 0) {
      console.warn("Warning: duration parameter can only be greater than zero if the left operand is variable!");
      this.duration = 0;
    }
  }

  evaluate(...eventParameters: any[]): boolean {
    const { lhsValue, rhsValue } = this.getOperands(...eventParameters);

    switch (this.operator) {
      case Operator.Equals:
      case Operator.NotEquals:
      case Operator.Greater:
      case Operator.Less:
      case Operator.GreaterOrEqual:
      case Operator.LessOrEqual:
        return this.compareOperands(lhsValue, this.operator, rhsValue);

      case Operator.Contains:
        return this.contains(lhsValue, rhsValue);

      case Operator.Regexp:
        return this.evalRegexp(lhsValue, rhsValue);
    }

    throw Error("Invalid Operator in condition!");
  }

  private compareOperands(lhsValue: ConditionOperandType, operator: Operator, rhsValue: ConditionOperandType): boolean {
    let durationSinceLastChange = 0;

    if (this.operandLhs instanceof VariableOperand) {
      durationSinceLastChange = performance.now() - this.operandLhs.lastModifiedMs;
    }

    if (lhsValue != undefined && rhsValue != undefined) {
      switch (operator) {
        case Operator.Equals:
          return (lhsValue == rhsValue && durationSinceLastChange >= this.duration) ? true : false;

        case Operator.NotEquals:
          return (lhsValue != rhsValue && durationSinceLastChange >= this.duration) ? true : false;

        case Operator.Greater:
          return (lhsValue > rhsValue && durationSinceLastChange >= this.duration) ? true : false;

        case Operator.Less:
          return (lhsValue < rhsValue && durationSinceLastChange >= this.duration) ? true : false;

        case Operator.GreaterOrEqual:
          return (lhsValue >= rhsValue && durationSinceLastChange >= this.duration) ? true : false;

        case Operator.LessOrEqual:
          return (lhsValue <= rhsValue && durationSinceLastChange >= this.duration) ? true : false;
      }
    }

    return false;
  }

  private contains(lhsValue: ConditionOperandType, rhsValue: ConditionOperandType): boolean {
    if (lhsValue != undefined && rhsValue != undefined) {
      return (lhsValue.toString().includes(rhsValue.toString())) ? true : false;
    } else {
      return false;
    }
  }

  private evalRegexp(lhsValue: ConditionOperandType, rhsValue: ConditionOperandType): boolean {
    if (lhsValue != undefined && rhsValue != undefined) {
      if (this.isValidRegularExpression(rhsValue.toString())) {
        const pattern = rhsValue.toString().split('/');
        const re = new RegExp(pattern[1], pattern[2]);
        const result = re.test(lhsValue.toString());
        return result ? true : false;
      } else {
        throw Error("Invalid regular experssion format!");
      }
    } else {
      return false;
    }
  }

  private getOperands(...eventParameters: any[]): {lhsValue: ConditionOperandType, rhsValue: ConditionOperandType} {
    if (this.operandLhs instanceof EventParameterOperand) {
      (this.operandLhs).setEventParameter(...eventParameters);
    }

    if (this.operandRhs instanceof EventParameterOperand) {
      (this.operandRhs).setEventParameter(...eventParameters);
    }

    const result = {
      lhsValue: this.operandLhs.value,
      rhsValue: this.operandRhs.value
    };

    return result;
  }

  private isValidRegularExpression(regexp: string): boolean {
    return /^([/~@;%#'])(.*?)\1([dgimsuy]*)$/.test(regexp);
  }
}
