const { Buffer } = require('buffer');
const xml2js = require('xml2js');

const httpStatusCodes = {
	100:"Continue",
	101:"Switching Protocols",
	200:"OK",
	201:"Created",
	202:"Accepted",
	203:"Non-Authoritative Information",
	204:"No Content",
	205:"Reset Content",
	206:"Partial Content",
	300:"Multiple Choices",
	301:"Moved Permanently",
	302:"Found",
	303:"See Other",
	304:"Not Modified",
	305:"Use Proxy",
	307:"Temporary Redirect",
	400:"Bad Request",
	401:"Unauthorized",
	402:"Payment Required",
	403:"Forbidden",
	404:"Not Found",
	405:"Method Not Allowed",
	406:"Not Acceptable",
	407:"Proxy Authentication Required",
	408:"Request Timeout",
	409:"Conflict",
	410:"Gone",
	411:"Length Required",
	412:"Precondition Failed",
	413:"Payload Too Large",
	414:"URI Too Long",
	415:"Unsupported Media Type",
	416:"Range Not Satisfiable",
	417:"Expectation Failed",
	426:"Upgrade Required",
	500:"Internal Server Error",
	501:"Not Implemented",
	502:"Bad Gateway",
	503:"Service Unavailable",
	504:"Gateway Time-out",
	505:"HTTP Version Not Supported",
	102:"Processing",
	207:"Multi-Status",
	226:"IM Used",
	308:"Permanent Redirect",
	422:"Unprocessable Entity",
	423:"Locked",
	424:"Failed Dependency",
	428:"Precondition Required",
	429:"Too Many Requests",
	431:"Request Header Fields Too Large",
	451:"Unavailable For Legal Reasons",
	506:"Variant Also Negotiates",
	507:"Insufficient Storage",
	511:"Network Authentication Required"
};

class RestClient {
	constructor(ipAddressOrHost, protocol, portNumber, authenticationType, username, password, customHeaders, certAuthEnable, logEnable = false, statusCodesToPass = undefined) {
		this.ipAddressOrHost = ipAddressOrHost;
		this.protocol = protocol;
		this.portNumber = portNumber;
		this.authenticationType = authenticationType;
		this.username = username;
		this.password = password;
		this.customHeaders = customHeaders;
		this.certAuthEnable = certAuthEnable;
		this.logEnable = logEnable || false;
		this.statusCodesToPass = statusCodesToPass || undefined;
	}

	get(path = '', parameters = '""', customHeaders = '{}') {
		if(path === null || path === undefined){
			path = "";
		}
		if(parameters === null || parameters === undefined){
			parameters = '""';
		}
		if(customHeaders === null || customHeaders === undefined){
			customHeaders = '{}';
		}
		try{
			customHeaders = JSON.parse(customHeaders);
			parameters = JSON.parse(parameters);
		}
		catch(e){
			return new Promise((resolve, reject)=>{reject("@get: Invalid 'Custom header' or 'Parameter' parameters!");});
		}
		return this.send(path, parameters, customHeaders, 'GET');
	}

	put(path = '', parameters = '""', customHeaders = '{}') {
		if(path === null || path === undefined){
			path = "";
		}
		if(parameters === null || parameters === undefined){
			parameters = '""';
		}
		if(customHeaders === null || customHeaders === undefined){
			customHeaders = '{}';
		}
		try{
			customHeaders = JSON.parse(customHeaders);
			parameters = JSON.parse(parameters);
		}
		catch(e){
			return new Promise((resolve, reject)=>{reject("@put: Invalid 'Custom header' or 'Parameter' parameters!");});
		}
		return this.send(path, parameters, customHeaders, 'PUT');
	}

	post(path = '', parameters = '""', customHeaders = '{}') {
		if(path === null || path === undefined){
			path = "";
		}
		if(parameters === null || parameters === undefined){
			parameters = '""';
		}
		if(customHeaders === null || customHeaders === undefined){
			customHeaders = '{}';
		}
		try{
			customHeaders = JSON.parse(customHeaders);
			parameters = JSON.parse(parameters);
		}
		catch(e){
			return new Promise((resolve, reject)=>{reject("@post: Invalid 'Custom header' or 'Parameter' parameters!");});
		}
		return this.send(path, parameters, customHeaders, 'POST');
	}

	delete(path = '', parameters = '""', customHeaders = '{}') {
		if(path === null || path === undefined){
			path = "";
		}
		if(parameters === null || parameters === undefined){
			parameters = '""';
		}
		if(customHeaders === null || customHeaders === undefined){
			customHeaders = '{}';
		}
		try{
			customHeaders = JSON.parse(customHeaders);
			parameters = JSON.parse(parameters);
		}
		catch(e){
			return new Promise((resolve, reject)=>{reject("@delete: Invalid 'Custom header' or 'Parameter' parameters!");});
		}
		return this.send(path, parameters, customHeaders, 'DELETE');
	}

	send(path, parameters, customHeaders, methodType){
		let options = {};
		switch(this.authenticationType){
			case "None":
				options = this.setOptions(methodType, path, parameters, customHeaders);
			break;
			case "Basic":
				options = this.setOptionsWithBasicAuthentication(methodType, path, parameters, customHeaders);
			break;
		}
		if (this.logEnable) {
			console.log(options);
		}
		return this.request(options, parameters);
	}

	request(options, parameters = null){
		return new Promise((resolve, reject) => {
			if(parameters != null){
				//If data to send is already string, do not stringify it, else, do that
				if(typeof parameters !== "string"){
					parameters = JSON.stringify(parameters);
				}

				if (!options.headers) {
					options.headers = {};
				}

				options.headers['Content-Length'] = parameters.length;
			}

			let req = {};
			if(/https/i.test(this.protocol)){
				if(!this.certAuthEnable) {
					options['rejectUnauthorized'] = false;
				}
				const https = require('https');
				req = https.request(options, (res) => {
					this.callbackFunction(res, resolve, reject);
				});
			} else{
				const http = require('http');
				req = http.request(options, (res) => {
					this.callbackFunction(res, resolve, reject);
				});
			}

			req.on('error', (e) => {
				reject(`Error with request: ${e.message}`);
			});

			if(parameters != null){
				if(this.logEnable){
					console.log(JSON.stringify(parameters));
				}

				// Write data to request body
				req.write(parameters);
			}
			req.end();
		});
	}

	callbackFunction(res, resolve, reject){
		res.parsedBodyContent = [];
		if (this.logEnable) {
			console.log('REQ. RESPONSE');
			console.log(`STATUS: ${res.statusCode}`);
			console.log(`HEADERS: ${JSON.stringify(res.headers)}`);
		}
		

		res.on('data', (chunk) => {
			res.parsedBodyContent.push(chunk);
		});

		res.on('end', async () => {
			if((200 <= res.statusCode && res.statusCode < 300) || (typeof this.statusCodesToPass != "undefined" && this.statusCodesToPass.indexOf(res.statusCode) != -1)){
				let body = await this.parseBody(res);
				resolve(body);
				if (this.logEnable) {
					console.log(`Response body: ${body}`);
				}
			}else{
				let errorPhrase = (typeof httpStatusCodes[res.statusCode] == "undefined" ? ("with status code:" + res.statusCode) : (res.statusCode + " - " +httpStatusCodes[res.statusCode]));
				reject(`Error: ${errorPhrase}`);
			}
		});
	}

	splitCustomHeader(){
		let paramsCustomHeadersObject = {};
		for( let k = 0; k < this.customHeaders.length; k++){
			if(this.customHeaders[k] == '') continue;
			let paramsCustomHeadersElement = this.customHeaders[k].split(":");
			paramsCustomHeadersObject[paramsCustomHeadersElement[0]] = paramsCustomHeadersElement[1];
		}

		return paramsCustomHeadersObject;
	}

	mergeHeaders(parameters, customHeaders){
		let splitCustomHeader = this.splitCustomHeader();
		let headers = {
		};
		if(typeof splitCustomHeader == "object" && Object.keys(splitCustomHeader).length > 0){
			Object.assign(headers, splitCustomHeader);
		}
		if(typeof customHeaders == "object" && Object.keys(customHeaders).length > 0){
			Object.assign(headers, customHeaders);
		}
		return headers;
	}
  
	setOptions(methodType, path, parameters, customHeaders){
		let headers = this.mergeHeaders(parameters, customHeaders);
		return {
			hostname: this.ipAddressOrHost,
			port: this.portNumber,
			path: path,
			method: methodType,
			headers: headers
		};
	}

	setOptionsWithBasicAuthentication(methodType, path, parameters, customHeaders){
		let headers = this.mergeHeaders(parameters, customHeaders);
		let auth = 'Basic ' + Buffer.from(this.username + ':' + this.password).toString('base64');
		headers['Authorization'] = auth;
		return {
			hostname: this.ipAddressOrHost,
			port: this.portNumber,
			path: path,
			method: methodType,
			headers: headers
		};
	}

	async parseBody(res){
		if (!res.parsedBodyContent.toString()) {
			return;
		}
		let type = this.getContentType(res.headers);
		switch(type){
			case "text":
				return res.parsedBodyContent.toString();
				break;
			case "json":
				return JSON.parse(res.parsedBodyContent.toString());
				break;
			case "xml":
				//Will return an object for ease
				let xmlStr = res.parsedBodyContent.toString();
				return xml2js.parseString(xmlStr, (err, result) => {
					return result;
				});
				break;
			case "binary":
				let binaryContent = Buffer.concat(res.parsedBodyContent);
				return binaryContent;
				break;
		}
	}

	getContentType(headers){
		let contentType = headers["content-type"];
		if(contentType){
			if(/^text\/*/i.test(contentType)){
				return "text";
			}
			if(/^application\/json/i.test(contentType)){
				return "json";
			}
			if(/^application\/xml/i.test(contentType)){
				return "xml";
			}
			if(/^application\/octet-stream/i.test(contentType)){
				return "binary";
			}
		}
		else{
			//If no supported format found, handle it as text content
			return "text";
		}
	}
}

exports.RestClient = RestClient;
