import {
  Condition,
  ConditionGroup,
  ConditionInterface,
  ConditionOperand,
  ConstOperand,
  EventParameterOperand,
  OperatorType,
  VariableOperand
} from "@lightware/condition-api";
import { Rule } from "./Rule";
import { Triggers } from "../Triggers/Triggers";
import { TriggerType } from "../Triggers/Trigger";
import { EventTrigger } from "../Triggers/EventTrigger";
import { VariableChangedIntoRangeTrigger, VariableChangedToValueTrigger } from "../Triggers/VariableTrigger";
import {
  ModuleDescriptor,
  RuleDescriptor,
  TriggerDescriptor,
  ConditionGroupDescriptor,
  ConditionDescriptor,
  LeftOperandDescriptor,
  RightOperandDescriptor,
  EventDescriptor,
  ConditionGroupOperator,
  toCronString
} from "@lightware/module-descriptor";
import TimeTrigger, { Options } from "../Triggers/TimeTrigger";
import VariableChangedTrigger from "../Triggers/VariableChangedTrigger";

type EventsWithInstanceId = Record<string, EventDescriptor[]>;

export class RuleManager {
  private rules: Rule[] = [];

  constructor(private readonly instanceApi: any) {}

  init(): void {
    const moduleDescriptorJSON = this.instanceApi.getModuleData(this.instanceApi.instanceDir);
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJSON);
    const events = this.getEvents(moduleDescriptor);
    this.rules = this.getRules(moduleDescriptor.rules, events);
  }

  setAction(ruleName: string, action: (...args: any[]) => Promise<void>): void {
    const rule = this.rules.find((r) => ruleName === r.name);
    if (rule) {
      rule.setAction(action);
    } else {
      console.warn(
        `Rule with name ${ruleName} not found when trying to set an action for it in instance with id: ${this.instanceApi.instanceId}`
      );
    }
  }

  /**
   * The goal is to return the events that is from the current instance or a remote instance.
   * @param moduleDescriptor
   * @returns A map with all the instances and its corresponding events.
   */
  private getEvents(moduleDescriptor: ModuleDescriptor): EventsWithInstanceId {
    let anotherInstanceIds = moduleDescriptor.rules.map((rule: RuleDescriptor) => {
      return rule.triggers.map((trigger: TriggerDescriptor) => {
        if (trigger.type === TriggerType.Event && trigger.details.instanceId !== "") {
          return trigger.details.instanceId;
        }
        return "";
      });
    }).flat();

    anotherInstanceIds = anotherInstanceIds
      .filter((element, index) => (anotherInstanceIds.indexOf(element) === index) && element !== "");

    const events: EventsWithInstanceId = { [this.instanceApi.instanceId]: moduleDescriptor.events };

    anotherInstanceIds.forEach((instanceId: string) => {
      const descriptorJSON = this.instanceApi.getModuleData(
        this.instanceApi.instanceDir.replace(this.instanceApi.instanceId, instanceId)
      );

      const moduleDescriptor = new ModuleDescriptor();
      moduleDescriptor.loadFromDescriptorObject(descriptorJSON);
      events[instanceId] = moduleDescriptor.events;
    });

    return events;
  }

  private getRules(rules: RuleDescriptor[], events: EventsWithInstanceId): Rule[] {
    return rules.map((rule: RuleDescriptor) => {
      const triggers = this.getTriggers(rule.triggers, events);
      const conditionGroup = rule.condition ? this.getConditionGroup(rule.condition, events) : undefined;

      return new Rule(rule.name, rule.enabled, triggers, conditionGroup);
    });
  }

  private getTriggers(
    moduleDescriptorTriggers: TriggerDescriptor[],
    events: EventsWithInstanceId
  ): Triggers {
    const triggers = new Triggers();

    moduleDescriptorTriggers.map((trigger: TriggerDescriptor) => {
      switch (trigger.type) {
        case (TriggerType.Event): {
          const eventTrigger = new EventTrigger(trigger.details.instanceId, trigger.details.event, this.instanceApi);
          triggers.addTrigger(eventTrigger);
          break;
        }

        case TriggerType.Time: {

          const options: Options = {
            cronTime: trigger.details.cron ? toCronString(trigger.details.cron) : undefined,
            startDate: trigger.details.startDate,
            endDate: trigger.details.endDate
          };

          const timeTrigger: TimeTrigger = new TimeTrigger(options);

          triggers.addTrigger(timeTrigger);

          break;
        }

        case (TriggerType.VariableChanged): {
          const instance = this.instanceApi.getInstanceById(trigger.details.instanceId);
          const remoteVariable = instance.variables[trigger.details.variable];

          const variableTrigger: VariableChangedTrigger = new VariableChangedTrigger(remoteVariable, trigger.duration);

          triggers.addTrigger(variableTrigger);
          break;
        }

        case (TriggerType.VariableChangedFromTo): {
          // TODO
          break;
        }

        case TriggerType.VariableChangedIntoRange:
        case (TriggerType.VariableChangedToValue): {
          const instance = this.instanceApi.getInstanceById(trigger.details.instanceId);
          const remoteVariable = instance.variables[trigger.details.variable];
          const conditionGroup = this.getConditionGroup(trigger.details.condition, events);

          if (!conditionGroup) {
            throw Error(
              'VariableChangedToValueTrigger descriptor is invalid. Descriptor has to contain at least one condition!.'
            );
          }

          let variableTrigger;
          if (trigger.type == TriggerType.VariableChangedToValue) {
            variableTrigger = new VariableChangedToValueTrigger(remoteVariable, conditionGroup, trigger.duration);
          } else {
            variableTrigger = new VariableChangedIntoRangeTrigger(remoteVariable, conditionGroup, trigger.duration);
          }

          triggers.addTrigger(variableTrigger);
          break;
        }
      }
    });

    return triggers;
  }

  private getConditionGroup(
    condition: ConditionGroupDescriptor | ConditionDescriptor,
    events: EventsWithInstanceId
  ): ConditionInterface | undefined {
    if (Object.keys(condition).length !== 0) {
      if (condition.type === 'condition') {
        const leftOperand = this.getOperand(condition.left, events);
        const rightOperand = this.getOperand(condition.right, events);
        return new Condition(leftOperand, condition.operator, rightOperand, condition.duration);
      } else {
        const operatorType = condition.operator === ConditionGroupOperator.AND ? OperatorType.And : OperatorType.Or;
        const conditionGroup = new ConditionGroup(operatorType);
        condition.conditions.map((cond) => {
          conditionGroup.addCondition(this.getConditionGroup(cond, events)!);
        });
        return conditionGroup;
      }
    } else {
      return undefined;
    }
  }

  private getOperand(
    operandDescriptor: LeftOperandDescriptor | RightOperandDescriptor,
    events: EventsWithInstanceId
  ): ConditionOperand {
    switch (operandDescriptor.type) {
      case 'eventCallbackParameter': {
        const event = events[operandDescriptor.instanceId].find((e) => e.name === operandDescriptor.event);
        if (event) {
          const index = event.getEventParameterIndex(operandDescriptor.name);
          return new EventParameterOperand(operandDescriptor.name, index);
        }
        throw new Error(
          `Could not find event with name ${operandDescriptor.event} on ${this.instanceApi.instanceId} instance`
        );
      }
      case 'StatusVariable': {
        return new VariableOperand(this.instanceApi, operandDescriptor.instanceId, operandDescriptor.name);
      }
      case 'Constant': {
        return new ConstOperand(operandDescriptor.value);
      }
      case 'InstanceParameter': {
        return new ConstOperand(this.instanceApi.params[operandDescriptor.name]);
      }
    }
    throw new Error(`Operand type: ${operandDescriptor.type}, not supported`);
  }
}
