import { RemoteVariable, TypeSignature } from "@lightware/variable-api";
import { VariableChangedMessage } from "@lightware/variable-api/dist/src/Communication/VariableChangedMessage";
import { VariableStateMessageData } from "@lightware/variable-api/dist/src/Communication/VariableStateMessageData";
import VariableChangedTrigger from "./VariableChangedTrigger";
import { timeEnd } from "console";

jest.useFakeTimers();

describe('VariableChanged trigger tests', () => {
  test('After variable changes the trigger has to be fired immediately.', () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const variableTrigger = new VariableChangedTrigger(remoteVariable);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
  });

  test('After variable changes the trigger has to be fired after 1 seconds.', () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const variableTrigger = new VariableChangedTrigger(remoteVariable, 1000);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    jest.advanceTimersByTime(1000);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
  });

  test('After variable changes the trigger has to be fired after 1 seconds. The event has to be fired only once', () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const variableTrigger = new VariableChangedTrigger(remoteVariable, 1000);

    let timeDifference = 0;

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);
    variableTrigger.onTriggered(() => {
      timeDifference = performance.now() - timeDifference
    });

    for(let i = 0; i < 5; i++) {
      timeDifference = performance.now();
      const currentValue = Math.ceil(Math.random() * 100);
      const data = new VariableStateMessageData("variable", '', TypeSignature.Number, currentValue , 21, timeDifference);
      const msg = new VariableChangedMessage('msgid', '', '', data);
      remoteVariable.onMessage(msg);
    }

    jest.advanceTimersByTime(1000);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
    expect(timeDifference).toEqual(1000);
  });
});