"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const VariableMessageType_1 = require("../Communication/VariableMessageType");
const VariableMessageBroker_1 = require("../VariableMessageBroker");
describe('VariableMesssageBroker', () => {
    let scriptrunner;
    let variableMesssageBroker;
    test('should create VariableMessageBroker instance successfully', () => {
        expect(() => {
            new VariableMessageBroker_1.VariableMesssageBroker(scriptrunner);
        }).not.toThrow();
        expect(new VariableMessageBroker_1.VariableMesssageBroker(scriptrunner)).toBeInstanceOf(VariableMessageBroker_1.VariableMesssageBroker);
    });
    describe('receiveMessage', () => {
        beforeEach(() => {
            jest.clearAllMocks();
            scriptrunner = {
                sendMessageToScript: jest.fn(),
            };
            variableMesssageBroker = new VariableMessageBroker_1.VariableMesssageBroker(scriptrunner);
        });
        test('should run without error with a valid JSON string', () => {
            const jsonMessage = JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            });
            expect(() => {
                variableMesssageBroker.receiveMessage(jsonMessage);
            }).not.toThrow();
        });
        test('should console.warn error if there is a wrong JSON string', () => {
            const consoleWarnSpy = jest.spyOn(global.console, 'warn');
            const jsonMessage = 'NOT a JSON string';
            variableMesssageBroker.receiveMessage(jsonMessage);
            expect(consoleWarnSpy).toHaveBeenCalled();
            expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
        });
        test('should add subscriber to a new subscription and send message to script with GetRemote messageType', () => {
            const jsonMessage = JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            });
            variableMesssageBroker.receiveMessage(jsonMessage);
            expect(scriptrunner.sendMessageToScript).toHaveBeenCalledWith('1', JSON.parse(jsonMessage));
            expect(variableMesssageBroker['subscriptionManager'].getSubscribers('1', 'cica')).toEqual(['subscriber1']);
        });
        test('should add new subscriber to subscription with GetRemote message type', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '2',
                source: 'subscriber2',
                messageData: {
                    variableName: 'cica2'
                }
            }));
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '2',
                source: 'subscriber3',
                messageData: {
                    variableName: 'cica2'
                }
            }));
            expect(variableMesssageBroker['subscriptionManager'].getSubscribers('2', 'cica2')).toEqual(['subscriber2', 'subscriber3']);
        });
        test('should not add the same subscriber to the same subscription with GetRemote message type', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            expect(variableMesssageBroker['subscriptionManager'].getSubscribers('1', 'cica')).toEqual(['subscriber1']);
        });
        test('should return empty array if .getSubscribers method does not found a subscriber', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(1);
            expect(variableMesssageBroker['subscriptionManager'].getSubscribers('2', 'cica')).toEqual([]);
        });
        test('should send message to script with GetRemoteResponse messageType', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemoteResponse,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(1);
        });
        test('should send message to all subscribers with Added messageType', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.Added,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(2);
        });
        test('should send message to all subscribers with Changed messageType', () => {
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.GetRemote,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            variableMesssageBroker.receiveMessage(JSON.stringify({
                messageType: VariableMessageType_1.VariableMessageType.Changed,
                instanceId: '1',
                source: 'subscriber1',
                messageData: {
                    variableName: 'cica'
                }
            }));
            expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(2);
        });
    });
});
//# sourceMappingURL=VariableMessageBroker.test.js.map