import { EventEmitter } from "events";
import { VariableMessage } from "./Communication/MessageParser";
import { VariableChangedMessage } from "./Communication/VariableChangedMessage";
import { VariableGetRemoteResponse } from "./Communication/VariableGetRemoteResponse";
import { StateChange, TypeSignature, Variable, VariableType } from "./Variable";
import { VariableAddedMessage } from "./Communication/VariableAddedMessage";
import { EventNames } from "./EventNames";
import { VariableStateMessageData } from "./Communication/VariableStateMessageData";

export class RemoteVariable implements Variable {
  public typeSignature: TypeSignature = TypeSignature.Undefined;

  private _currentValue: VariableType = undefined;
  private _previousValue: VariableType = undefined;

  private _lastModifiedMs: number = -1;

  private readonly eventEmitter: NodeJS.EventEmitter = new EventEmitter();

  constructor(private readonly name: string, private readonly _ownerId: string) {}

  isValid(): boolean {
    return this._currentValue !== undefined;
  }

  get ownerId(): string {
    return this._ownerId;
  }

  get value(): VariableType {
    return this._currentValue;
  }

  set value(_value: VariableType) {
    const warnMessage1 = `The value of LARA variable ${this.name} (owned by instance ${this._ownerId}) can not be set .`;
    const warnMessage2 = `The value of a LARA variable can only be set by their owner.`;
    console.warn(`${warnMessage1} ${warnMessage2}`);
  }

  get previousValue(): VariableType {
    return this._previousValue;
  }

  get lastModifiedMs(): number {
    return this._lastModifiedMs;
  }

  onValid(listener: () => void): void {
    this.eventEmitter.on(EventNames.Valid, listener)
  }

  onInvalid(listener: () => void): void {
    this.eventEmitter.on(EventNames.Invalid, listener);
  }

  onChanged(listener: (currentValue: VariableType, previousValue: VariableType) => void): void {
    this.eventEmitter.on(EventNames.Changed, listener);
  }

  onMessage(message: VariableMessage): void {
    if (message instanceof VariableAddedMessage ||
        message instanceof VariableChangedMessage ||
        message instanceof VariableGetRemoteResponse) {

      this.setInternalState(message.messageData);
    }
  }

  private setLastModifiedMs(time: number): void {
    if (this._lastModifiedMs > time) {
      console.warn(`The incoming timestamp (${time} ms) is earlier then the current timestamp (${this._lastModifiedMs} ms) in the ${this._ownerId}.${this.name} status variable!`);
    }

    this._lastModifiedMs = time;
  }

  private setInternalState(messageData: VariableStateMessageData): void {
    if (this._currentValue != messageData.currentValue) {
      let stateChange: StateChange = StateChange.None;

      if (this.isValid() === false && messageData.currentValue !== undefined) {
        stateChange = StateChange.ToValid;
      } else if(this.isValid() === true && messageData.currentValue === undefined) {
        stateChange = StateChange.ToInvalid;
      }

      this._currentValue = messageData.currentValue;
      this._previousValue = messageData.previousValue;

      this.typeSignature = messageData.typeSignature!;

      this.setLastModifiedMs(messageData.lastModifiedMs!);

      if (stateChange == StateChange.ToValid) {
        this.eventEmitter.emit(EventNames.Valid);
      } else if (stateChange == StateChange.ToInvalid) {
        this.eventEmitter.emit(EventNames.Invalid);
      }

      this.eventEmitter.emit(EventNames.Changed, this._currentValue, this._previousValue);
    }
  }
}
