import { VariableAddedMessage } from "../Communication/VariableAddedMessage";
import { VariableChangedMessage } from "../Communication/VariableChangedMessage";
import { VariableGetRemoteResponse } from "../Communication/VariableGetRemoteResponse";
import { VariableMessageType } from "../Communication/VariableMessageType";
import { RemoteVariable } from "../RemoteVariable";

describe('RemoteVariable', () => {
  afterEach(() => {
    jest.clearAllMocks();
  });

  const consoleWarnSpy = jest.spyOn(console, 'warn').mockImplementationOnce(() => {});

  test('setLastModifiedMs check if current last modified time is less then previous one', () => {
    const remoteVariable = new RemoteVariable("cica", "taurus");

    expect(remoteVariable.value).toBe(undefined);

    const jsonMessage: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const json = JSON.parse(JSON.stringify(jsonMessage));
    const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    const jsonMessage2: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 60,
        previousValue: 10,
        lastModifiedMs: 12
      }
    }
    const json2 = JSON.parse(JSON.stringify(jsonMessage2));
    const convertedMessage2 = Object.setPrototypeOf(json2, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage2);

    expect(remoteVariable.value).toBe(60);
    expect(remoteVariable.previousValue).toBe(10);
    expect(remoteVariable.lastModifiedMs).toBe(12);
    expect(consoleWarnSpy).toBeCalledTimes(1);
    expect(consoleWarnSpy).toBeCalledWith(`The incoming timestamp (12 ms) is earlier then the current timestamp (123 ms) in the taurus.cica status variable!`);
  });

  test('Successfully create a RemoteVariable', () => {
    expect(() => {
      new RemoteVariable("cica", "ownerID");
    }).not.toThrow();
  });

  test('Cannot set remoteVariable', () => {
    const remoteVariable = new RemoteVariable("cica", "ownerID");
    expect(() => {
      remoteVariable.value = 10;
    }).not.toThrow();

    expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
    expect(consoleWarnSpy).toHaveBeenLastCalledWith(
      "The value of LARA variable cica (owned by instance ownerID) can not be set . The value of a LARA variable can only be set by their owner."
    );
  });

  test('Check remoteVariable is valid', () => {
    const remoteVariable = new RemoteVariable("cica", "ownerID");
    expect(remoteVariable.isValid()).toBeFalsy();
  });

  test('get ownerID', () => {
    const remoteVariable = new RemoteVariable("cica", "ownerID");
    expect(remoteVariable.ownerId).toBe("ownerID");
  });

  test('onValid event called with onMessage(VariableChangedMessage)', () => {
    const remoteVariable = new RemoteVariable("cica", "taurus");
    const onValidHandler = jest.fn();

    remoteVariable.onValid(onValidHandler);

    expect(remoteVariable.value).toBe(undefined);

    const jsonMessage: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const json = JSON.parse(JSON.stringify(jsonMessage));
    const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    expect(onValidHandler).toHaveBeenCalledTimes(1);
    expect(remoteVariable.value).toBe(10);
    expect(remoteVariable.previousValue).toBe(20);
    expect(remoteVariable.lastModifiedMs).toBe(123);
  });

  test('onInvalid event called', () => {
    const remoteVariable = new RemoteVariable("cica", "taurus");
    const onInvalidHandler = jest.fn();

    remoteVariable.onInvalid(onInvalidHandler);

    expect(remoteVariable.value).toBe(undefined);

    const jsonMessage: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const jsonMessage2: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: undefined,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    let json = JSON.parse(JSON.stringify(jsonMessage));
    let convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    json = JSON.parse(JSON.stringify(jsonMessage2));
    convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    expect(onInvalidHandler).toHaveBeenCalledTimes(1);
  });

  test('onChanged event called when variable changed, but not calledd when subscricption occurs.', () => {
    const remoteVariable = new RemoteVariable("cica", "taurus");
    const onChangedHandler = jest.fn();

    remoteVariable.onChanged(onChangedHandler);

    expect(remoteVariable.value).toBe(undefined);

    const jsonMessage: VariableChangedMessage = {
      messageId: 'msg1',
      instanceId: 'taurus',
      source: 'source',
      messageType: VariableMessageType.Changed,
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const json = JSON.parse(JSON.stringify(jsonMessage));
    const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    expect(onChangedHandler).toHaveBeenCalledTimes(1);
    expect(remoteVariable.value).toBe(10);
    expect(remoteVariable.previousValue).toBe(20);
    expect(remoteVariable.lastModifiedMs).toBe(123);
  });

  test('onMessage event called with VariableGetRemoteResponse', () => {
    const setInternalStateSpy = jest.spyOn(RemoteVariable.prototype as any, 'setInternalState').mockImplementation(jest.fn());
    const remoteVariable = new RemoteVariable("cica", "taurus");

    const jsonMessage: VariableGetRemoteResponse = {
      messageType: VariableMessageType.GetRemoteResponse,
      instanceId: 'taurus',
      source: 'subscriber1',
      messageId: 'msg1',
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const json = JSON.parse(JSON.stringify(jsonMessage));
    const convertedMessage = Object.setPrototypeOf(json, VariableGetRemoteResponse.prototype)
    remoteVariable.onMessage(convertedMessage);

    expect(setInternalStateSpy).toHaveBeenCalledTimes(1);
  });

  test('onMessage event called with VariableAddedMessage', () => {
    const setInternalStateSpy = jest.spyOn(RemoteVariable.prototype as any, 'setInternalState').mockImplementation(jest.fn());
    const remoteVariable = new RemoteVariable("cica", "taurus");

    const jsonMessage: VariableAddedMessage = {
      messageType: VariableMessageType.Added,
      instanceId: 'taurus',
      source: 'subscriber1',
      messageId: 'msg1',
      messageData: {
        variableName: 'cica',
        ownerId: 'taurus',
        currentValue: 10,
        previousValue: 20,
        lastModifiedMs: 123
      }
    }

    const json = JSON.parse(JSON.stringify(jsonMessage));
    const convertedMessage = Object.setPrototypeOf(json, VariableAddedMessage.prototype)
    remoteVariable.onMessage(convertedMessage);

    expect(setInternalStateSpy).toHaveBeenCalledTimes(1);
  });
});