"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Condition = exports.createCondition = exports.Operator = void 0;
const ConstOperand_1 = require("./ConditionOperands/ConstOperand");
const EventParameterOperand_1 = require("./ConditionOperands/EventParameterOperand");
const VariableOperand_1 = require("./ConditionOperands/VariableOperand");
var Operator;
(function (Operator) {
    Operator["Equals"] = "equals";
    Operator["NotEquals"] = "not_equals";
    Operator["Less"] = "less";
    Operator["LessOrEqual"] = "less_or_equal";
    Operator["Greater"] = "greater";
    Operator["GreaterOrEqual"] = "greater_or_equal";
    Operator["Contains"] = "contains";
    Operator["Regexp"] = "regexp";
    Operator["=="] = "equals";
    Operator["!="] = "not_equals";
    Operator["<"] = "less";
    Operator["<="] = "less_or_equal";
    Operator[">"] = "greater";
    Operator[">="] = "greater_or_equal";
    Operator["contains"] = "contains";
    Operator["regexp"] = "regexp";
})(Operator = exports.Operator || (exports.Operator = {}));
/* eslint-disable */
(function (Operator) {
    function isValidKey(value) {
        return Object.keys(Operator).includes(value);
    }
    Operator.isValidKey = isValidKey;
    function isValidValue(value) {
        return Object.values(Operator).includes(value);
    }
    Operator.isValidValue = isValidValue;
})(Operator = exports.Operator || (exports.Operator = {}));
/* eslint-enable */
function createCondition(operandLhs, operator, operandRhs, duration = 0) {
    if (typeof operandRhs === 'number' || typeof operandRhs === 'string' || typeof operandRhs === 'boolean') {
        const value = operandRhs;
        operandRhs = new ConstOperand_1.ConstOperand(value);
    }
    return new Condition(operandLhs, operator, operandRhs, duration);
}
exports.createCondition = createCondition;
class Condition {
    constructor(operandLhs, operator, operandRhs, duration = 0) {
        this.operandLhs = operandLhs;
        this.operator = operator;
        this.operandRhs = operandRhs;
        this.duration = duration;
        if (!Operator.isValidValue(operator)) {
            if (Operator.isValidKey(operator)) {
                this.operator = Operator[operator];
            }
            else {
                throw Error(`Invalid conditional operator of ${this.operator}`);
            }
        }
        if (this.operator == Operator.Regexp && !(operandRhs instanceof ConstOperand_1.ConstOperand)) {
            throw Error("Right Handside operand must be 'ConstOperand', if the operator is 'Regexp'");
        }
        if (!(this.operandLhs instanceof VariableOperand_1.VariableOperand) && this.duration > 0) {
            console.warn("Warning: duration parameter can only be greater than zero if the left operand is variable!");
            this.duration = 0;
        }
    }
    evaluate(...eventParameters) {
        const { lhsValue, rhsValue } = this.getOperands(...eventParameters);
        switch (this.operator) {
            case Operator.Equals:
            case Operator.NotEquals:
            case Operator.Greater:
            case Operator.Less:
            case Operator.GreaterOrEqual:
            case Operator.LessOrEqual:
                return this.compareOperands(lhsValue, this.operator, rhsValue);
            case Operator.Contains:
                return this.contains(lhsValue, rhsValue);
            case Operator.Regexp:
                return this.evalRegexp(lhsValue, rhsValue);
        }
        throw Error("Invalid Operator in condition!");
    }
    compareOperands(lhsValue, operator, rhsValue) {
        let durationSinceLastChange = 0;
        if (this.operandLhs instanceof VariableOperand_1.VariableOperand) {
            durationSinceLastChange = performance.now() - this.operandLhs.lastModifiedMs;
        }
        if (lhsValue != undefined && rhsValue != undefined) {
            switch (operator) {
                case Operator.Equals:
                    return (lhsValue == rhsValue && durationSinceLastChange >= this.duration) ? true : false;
                case Operator.NotEquals:
                    return (lhsValue != rhsValue && durationSinceLastChange >= this.duration) ? true : false;
                case Operator.Greater:
                    return (lhsValue > rhsValue && durationSinceLastChange >= this.duration) ? true : false;
                case Operator.Less:
                    return (lhsValue < rhsValue && durationSinceLastChange >= this.duration) ? true : false;
                case Operator.GreaterOrEqual:
                    return (lhsValue >= rhsValue && durationSinceLastChange >= this.duration) ? true : false;
                case Operator.LessOrEqual:
                    return (lhsValue <= rhsValue && durationSinceLastChange >= this.duration) ? true : false;
            }
        }
        return false;
    }
    contains(lhsValue, rhsValue) {
        if (lhsValue != undefined && rhsValue != undefined) {
            return (lhsValue.toString().includes(rhsValue.toString())) ? true : false;
        }
        else {
            return false;
        }
    }
    evalRegexp(lhsValue, rhsValue) {
        if (lhsValue != undefined && rhsValue != undefined) {
            if (this.isValidRegularExpression(rhsValue.toString())) {
                const pattern = rhsValue.toString().split('/');
                const re = new RegExp(pattern[1], pattern[2]);
                const result = re.test(lhsValue.toString());
                return result ? true : false;
            }
            else {
                throw Error("Invalid regular experssion format!");
            }
        }
        else {
            return false;
        }
    }
    getOperands(...eventParameters) {
        if (this.operandLhs instanceof EventParameterOperand_1.EventParameterOperand) {
            (this.operandLhs).setEventParameter(...eventParameters);
        }
        if (this.operandRhs instanceof EventParameterOperand_1.EventParameterOperand) {
            (this.operandRhs).setEventParameter(...eventParameters);
        }
        const result = {
            lhsValue: this.operandLhs.value,
            rhsValue: this.operandRhs.value
        };
        return result;
    }
    isValidRegularExpression(regexp) {
        return /^([/~@;%#'])(.*?)\1([dgimsuy]*)$/.test(regexp);
    }
}
exports.Condition = Condition;
//# sourceMappingURL=Condition.js.map