import { Condition, Operator, createCondition } from "./Condition";
import { ConditionGroup, OperatorType } from "./ConditionGroup";
import { ConstOperand } from "./ConditionOperands/ConstOperand";
import { EventParameterOperand } from "./ConditionOperands/EventParameterOperand";
import { VariableOperand } from "./ConditionOperands/VariableOperand";

// It is need for nodejs version 16, because the now property is readonly.
Object.defineProperty(performance, "now", {
  value: jest.fn(),
  configurable: true,
  writable: true
});

describe('Operator enum', () => {
  test('checks if a value is a valid key', () => {
    expect(Operator.isValidKey('==')).toBe(true);
    expect(Operator.isValidKey('!=')).toBe(true);
    expect(Operator.isValidKey('<')).toBe(true);
    expect(Operator.isValidKey('<=')).toBe(true);
    expect(Operator.isValidKey('>')).toBe(true);
    expect(Operator.isValidKey('>=')).toBe(true);
    expect(Operator.isValidKey('contains')).toBe(true);
    expect(Operator.isValidKey('regexp')).toBe(true);
    expect(Operator.isValidKey('invalid')).toBe(false);
  });

  test('checks if a value is a valid operator', () => {
    expect(Operator.isValidValue('equals')).toBe(true);
    expect(Operator.isValidValue('not_equals')).toBe(true);
    expect(Operator.isValidValue('less')).toBe(true);
    expect(Operator.isValidValue('less_or_equal')).toBe(true);
    expect(Operator.isValidValue('greater')).toBe(true);
    expect(Operator.isValidValue('greater_or_equal')).toBe(true);
    expect(Operator.isValidValue('contains')).toBe(true);
    expect(Operator.isValidValue('regexp')).toBe(true);
    expect(Operator.isValidValue('invalid')).toBe(false);
  });
});

describe('createCondition function', () => {
  test('should create a Condition with Constant operands', () => {
    const operandLhs = new ConstOperand(5);
    const operator = Operator.Equals;
    const operandRhs = new ConstOperand(5);
    const condition = createCondition(operandLhs, operator, operandRhs);

    expect(condition).toBeInstanceOf(Condition);
    expect(condition['operandLhs']).toBe(operandLhs);
    expect(condition['operator']).toBe(operator);
    expect(condition['operandRhs']).toBe(operandRhs);

  });

  test('should create a Condition with Variable operands', () => {
    const instanceApiMockL = {
      getInstanceById: jest.fn().mockImplementation(() => {
        return {
          variables: {
            test: {
              value: 10,
              lastModifiedMs: 1000,
              onChanged: jest.fn()
            }
          }
        };
      })
    };
    const operandLhs = new VariableOperand(instanceApiMockL, 'test', 'test');

    const instanceApiMockR = {
      getInstanceById: jest.fn().mockImplementation(() => {
        return {
          variables: {
            test: {
              value: 5,
              lastModifiedMs: 2000,
              onChanged: jest.fn()
            }
          }
        };
      })
    };
    const operandRhs = new VariableOperand(instanceApiMockR, 'test', 'test');
    const operator = Operator.Greater;
    const condition = createCondition(operandLhs, operator, operandRhs);

    expect(condition).toBeInstanceOf(Condition);
    expect(condition['operandLhs']).toBe(operandLhs);
    expect(condition['operator']).toBe(operator);
    expect(condition['operandRhs']).toBe(operandRhs);
  });

  test('should create a Condition with a numeric RHS value', () => {
    const operandLhs = new ConstOperand(5);
    const operator = Operator.Less;
    const operandRhs = 10;
    const condition = createCondition(operandLhs, operator, operandRhs);

    expect(condition).toBeInstanceOf(Condition);
    expect(condition['operandLhs']).toBe(operandLhs);
    expect(condition['operator']).toBe(operator);
    expect(condition['operandRhs']).toBeInstanceOf(ConstOperand);
    expect(condition['operandRhs']['value']).toBe(operandRhs);
  });

  test('should create a Condition with a string RHS value', () => {
    const operandLhs = new ConstOperand(5);
    const operator = Operator.Less;
    const operandRhs = 'string';
    const condition = createCondition(operandLhs, operator, operandRhs);

    expect(condition).toBeInstanceOf(Condition);
    expect(condition['operandLhs']).toBe(operandLhs);
    expect(condition['operator']).toBe(operator);
    expect(condition['operandRhs']).toBeInstanceOf(ConstOperand);
    expect(condition['operandRhs']['value']).toBe(operandRhs);
  });

  test('should create a Condition with a boolean RHS value', () => {
    const operandLhs = new ConstOperand(5);
    const operator = Operator.Less;
    const operandRhs = true;
    const condition = createCondition(operandLhs, operator, operandRhs);

    expect(condition).toBeInstanceOf(Condition);
    expect(condition['operandLhs']).toBe(operandLhs);
    expect(condition['operator']).toBe(operator);
    expect(condition['operandRhs']).toBeInstanceOf(ConstOperand);
    expect(condition['operandRhs']['value']).toBe(operandRhs);
  });
});

describe('Condition Class Test', () => {
  const eventParameters = [
    10,
    "foo",
    true,
    { "name": "John", "age": 30, "car": null }
  ];

  test('should throw error if the operator is invalid', () => {
    expect(() => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);
      conditionGroupAnd.addCondition(new Condition(new ConstOperand(10), "====", new ConstOperand(10)));
    }).toThrow("Invalid conditional operator of ====");
  });

  test('should throw error if rhs is not a ConstOperand', () => {
    const conditionGroupAnd = new ConditionGroup(OperatorType.And);
    expect(() => {
      conditionGroupAnd.addCondition(
        new Condition(
          new ConstOperand("Hello World"),
          Operator.Regexp,
          new EventParameterOperand("arg", 0)
        )
      );
    }).toThrow("Right Handside operand must be 'ConstOperand', if the operator is 'Regexp'");
  });

  test("should warn and set duration to 0 if lhs operand is not VariableOperand and duration > 0", () => {
    console.warn = jest.fn();
    const operandLhs = new ConstOperand(5);
    const operandRhs = new ConstOperand(10);
    const condition = new Condition(operandLhs, "equals", operandRhs, 100);
    expect(console.warn).toHaveBeenCalledWith("Warning: duration parameter can only be greater than zero if the left operand is variable!");
    expect(condition['duration']).toBe(0);
  });

  test('should return false if Eventparameter index is invalid', () => {
    const conditionGroupAnd = new ConditionGroup(OperatorType.And);

    expect(() => {
      conditionGroupAnd.addCondition(
        new Condition(
          new EventParameterOperand("invalidName", 5),
          "==",
          new ConstOperand(10)
        )
      );
    }).not.toThrow();

    expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
  });

  describe('compareOperands', () => {
    const operatorEqualArr: Operator[] = [Operator.Equals, Operator.LessOrEqual, Operator.GreaterOrEqual];
    const operatorLessArr: Operator[] = [Operator.NotEquals, Operator.Less, Operator.LessOrEqual];
    const operatorGreaterArr: Operator[] = [Operator.Greater, Operator.GreaterOrEqual];

    let instanceApiMock: /* eslint-disable */ any /* eslint-enable */;
    let operandLhs: VariableOperand;
    const duration = 500;

    beforeEach(() => {
      instanceApiMock = {
        getInstanceById: jest.fn().mockImplementation(() => {
          return {
            variables: {
              test: {
                value: 5,
                lastModifiedMs: 1000,
                onChanged: jest.fn()
              }
            }
          };
        })
      };
      operandLhs = new VariableOperand(instanceApiMock, 'test', 'test');
    });

    afterEach(() => {
      jest.clearAllMocks();
    });

    operatorEqualArr.forEach((operator) => {
      test(`should return true with ${operator} operator, when lhsValue is equals to rhsValue, and durationSinceLastChange is greater than duration`, () => {
        const operandRhs = new ConstOperand(5);
        const currentMs = 2000;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(true);
      });

      test(`should return false with ${operator} operator, when lhsValue is equals to rhsValue, and durationSinceLastChange is less than duration`, () => {
        const operandRhs = new ConstOperand(5);
        const currentMs = 1400;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(false);
      });
    });

    test(`should return false with Equals operator, when lhsValue is NOT equals to rhsValue, and durationSinceLastChange is greater than duration`, () => {
      const operandRhs = new ConstOperand(10);
      const currentMs = 2000;
      jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
      const condition = new Condition(operandLhs, Operator.Equals, operandRhs, duration);
      const result = condition.evaluate(operandLhs.value, Operator.Equals, operandRhs.value);

      expect(result).toBe(false);
    });

    operatorLessArr.forEach((operator) => {
      test(`should return true with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is greater than duration`, () => {
        const operandRhs = new ConstOperand(10);
        const currentMs = 2000;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(true);
      });

      test(`should return false with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is less than duration`, () => {
        const operandRhs = new ConstOperand(10);
        const currentMs = 1400;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(false);
      });
      if (operator != Operator.NotEquals) {
        test(`should return false with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is greater than duration`, () => {
          const operandRhs = new ConstOperand(1);
          const currentMs = 2000;
          jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
          const condition = new Condition(operandLhs, operator, operandRhs, duration);
          const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

          expect(result).toBe(false);
        });
      }
    });

    operatorGreaterArr.forEach((operator) => {
      test(`should return true with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is greater than duration`, () => {
        const operandRhs = new ConstOperand(1);
        const currentMs = 2000;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(true);
      });

      test(`should return false with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is less than duration`, () => {
        const operandRhs = new ConstOperand(1);
        const currentMs = 1400;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(false);
      });

      test(`should return false with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is greater than duration`, () => {
        const operandRhs = new ConstOperand(10);
        const currentMs = 2000;
        jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
        const condition = new Condition(operandLhs, operator, operandRhs, duration);
        const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);

        expect(result).toBe(false);
      });
    });
  });

  describe('evaluate', () => {
    test('should return true if number == number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            Operator.Equals,
            new ConstOperand(10)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            Operator.Equals,
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if number == string', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "==",
            new ConstOperand("10")
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
    });

    test('should return true if string == number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new ConstOperand("10"),
            "==",
            new EventParameterOperand("valNumber", 0)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
    });

    test('should return true if number < number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "<",
            new ConstOperand(11)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "<",
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if number > number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            ">",
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            ">",
            new ConstOperand(11)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if number != number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "!=",
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "!=",
            new ConstOperand(10)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if number >= number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            ">=",
            new ConstOperand(10)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            ">=",
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            ">=",
            new ConstOperand(11)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if number <= number', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "<=",
            new ConstOperand(10)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "<=",
            new ConstOperand(11)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valNumber", 0),
            "<=",
            new ConstOperand(9)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });

    test('should return true if value == true', () => {
      const conditionGroupAnd = new ConditionGroup(OperatorType.And);

      expect(() => {
        conditionGroupAnd.addCondition(
          new Condition(
            new EventParameterOperand("valBool", 2),
            "==",
            new ConstOperand(true)
          )
        );
      }).not.toThrow();

      expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
    });

    describe('String Contains', () => {
      test('should return true if string contains string', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          conditionGroupAnd.addCondition(
            new Condition(
              new ConstOperand("Hello World"),
              Operator.Contains,
              new ConstOperand("llo W")
            )
          );
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(true);
      });

      test('should return false if string not contains string', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          const condition = new Condition(
            new ConstOperand("Hello World"),
            Operator.Contains,
            new ConstOperand("notContains")
          );

          conditionGroupAnd.addCondition(condition);
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(false);
      });

      test('should return false if string not contains string', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          const condition = new Condition(new ConstOperand("Hello World"), 'contains', new ConstOperand("notContains"));

          conditionGroupAnd.addCondition(condition);
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(false);
      });

      test('should return true if string contains a number', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(async () => {
          const condition = new Condition(new ConstOperand("Hello 10  World"), Operator.Contains, new ConstOperand(10));
          conditionGroupAnd.addCondition(condition);
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(true);
      });
    });

    describe('Regexps', () => {
      test('should return false if pattern is invalid', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          conditionGroupAnd.addCondition(
            new Condition(
              new ConstOperand("Hello World"),
              Operator.Regexp,
              new ConstOperand('invalid')
            )
          );
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(false);
      });

      test('Regexp w/o flags', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          conditionGroupAnd.addCondition(
            new Condition(
              new ConstOperand("Hello World"),
              Operator.Regexp,
              new ConstOperand("/^H.*d/")
            )
          );
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(true);
      });

      test('Regexp w/o, false output', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          conditionGroupAnd.addCondition(
            new Condition(
              new ConstOperand("Hello World"),
              Operator.Regexp,
              new ConstOperand("/^H.*D/")
            )
          );
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(false);
      });

      test('Regexp with flags', () => {
        const conditionGroupAnd = new ConditionGroup(OperatorType.And);

        expect(() => {
          conditionGroupAnd.addCondition(
            new Condition(
              new ConstOperand("Hello World"),
              Operator.Regexp,
              new ConstOperand("/^h.*d/i")
            )
          );
        }).not.toThrow();

        expect(conditionGroupAnd.evaluate([])).toBe(true);
      });
    });
  });
});
