const common = require('./common');
const variableApi = require('@lightware/variable-api');

class EventAPI {
  constructor(props) {
    this.comm = props.comm;
    this.subscriptions = [];
  }

  on(event, callback) {
    if (typeof this.subscriptions[event] == 'undefined') {
      this.subscriptions[event] = [];
    }

    this.subscriptions[event].push(callback);

    this.currentMessageId++;

    this.comm.send({
      messageId: this.instanceId + '-' + this.currentMessageId,
      instanceId: this.instanceId,
      type: 'subscribeToEvent',
      event: event
    });
  }

  emit(event, ...args) {
    if (typeof this.subscriptions[event] != 'undefined') {
      this.subscriptions[event].forEach(callback => {
        if (typeof callback == 'function') {
          callback.apply(null, args);
        }
      });
    }
  }
}

/**
 * Instanciated by InstanceApi.getInstanceById method
 * Returns an interface to a remote LARA module instance
 */
class InstanceInterface extends EventAPI {
  currentMessageId = 0;
  messageStore = {};
  variables = {};

  constructor(props) {
    super(props);

    this.parent = props.parent;

    this.instanceId = props.instanceId;
    this.moduleDescriptor = props.moduleDescriptor;

    this.RemoteVariableManager = new variableApi.RemoteVariableManager(this.instanceId, this.parent.instanceId, this.comm);
    this.variables = this.RemoteVariableManager.proxy;

    this.generateInterface();

    this.initMessageChannelListener();
  }

  initMessageChannelListener() {
    this.comm.on('message', async data => {
      this.RemoteVariableManager.receiveMessage(data);
      if (typeof data == 'string') {
        data = JSON.parse(data);
      }

      switch (data.type) {
        case 'dispatchEvent':
          if (data.source && this.instanceId == data.source) {
            this.emit(data.event, ...(data.args || []));
          }
          break;
        case 'result':
          if (typeof this.messageStore[data.messageId] != 'undefined') {
            if (data.resultType == 'resolve') {
              if (/function/i.test(typeof this.messageStore[data.messageId].resolve)) {
                this.messageStore[data.messageId].resolve(data.result);
              }
            } else if (data.resultType == 'reject') {
              if (/function/i.test(typeof this.messageStore[data.messageId].reject)) {
                this.messageStore[data.messageId].reject(data.result);
              }
            }

            delete this.messageStore[data.messageId];
          }
          break;
      }
    });
  }

  createVirtualMethods(items) {
    for (const [key, value] of Object.entries(items)) {
      if (value.type == 'method') {
        if (typeof this[key] == 'undefined') {
          this[key] = (...args) => {
            this.currentMessageId++;
            const fullMessageId = this.instanceId + '-' + this.currentMessageId;

            this.messageStore[fullMessageId] = {};
            this.messageStore[fullMessageId].promise =
              new Promise((resolve, reject) => {
                this.messageStore[fullMessageId].resolve = resolve;
                this.messageStore[fullMessageId].reject = reject;

                const res = common.checkArguments(value.parameters, args);

                if (res.success) {
                  this.comm.send({
                    messageId: fullMessageId,
                    instanceId: this.instanceId,
                    type: 'callMethod',
                    methodName: key,
                    params: res.args
                  });
                } else {
                  reject(new Error(res.errorMsg));
                }
              });

            return this.messageStore[fullMessageId].promise;
          };
        }
      }
    }
  }

  createVariables(items) {
    for (const [key, value] of Object.entries(items)) {
      if (value.type == 'variable') {
        this.RemoteVariableManager.createVariablePlaceholder(key);
      }
    }
  }

  generateInterface() {
    if (this.moduleDescriptor.Devices) {
      for (const [devicekey, device] of Object.entries(this.moduleDescriptor.Devices)) {
        if (device.children) {
          this.createVirtualMethods(device.children);
          this.createVariables(device.children);
        }

        if (device.extensions) {
          this.createVirtualMethods(device.extensions);
          this.createVariables(device.extensions);
        }
      }
    }
  }
}

exports.InstanceInterface = InstanceInterface;
