import { ActionsDescriptor } from './ActionsDescriptor';
import { MethodDescriptor } from './MethodDescriptor';
import { ModuleDescriptor } from './ModuleDescriptor';
import fs from 'fs/promises';
jest.mock('fs/promises');

describe('ModuleDescriptor', () => {
  const testParameter = {
    name: 'testParam',
    title: 'test',
    label: 'test',
    description: 'test',
    type: 'string' as any
  };
  const testActions = {
    type: 'code',
    steps: [],
    code: ''
  };
  const moduleDescriptorJson = {
    Schemas: {},
    Devices: {
      Test: {
        name: 'testModule',
        type: 'group',
        children: {
          testMethod1: {
            parameters: [testParameter],
            actions: testActions,
            description: '',
            returnType: '',
            orderId: 0,
            type: 'method'
          },
          testVariable1: {
            valueType: 'string',
            defaultValue: 'test',
            description: 'test',
            orderId: 0,
            type: 'variable'
          },
          testEvent1: {
            callbackParameters: [testParameter],
            actions: testActions,
            description: 'test',
            subscriptionMethodCode: 'test',
            orderId: 0,
            baseEventTemplateName: 'test',
            parameterValues: { test: 'test' },
            type: 'event'
          }
        },
        extensions: {
          testMethod2: {
            parameters: [testParameter],
            actions: testActions,
            description: '',
            returnType: '',
            orderId: 1,
            type: 'method'
          },
          testVariable2: {
            valueType: 'string',
            defaultValue: 'test',
            description: 'test',
            orderId: 1,
            type: 'variable'
          },
          testEvent2: {
            callbackParameters: [testParameter],
            actions: testActions,
            description: 'test',
            subscriptionMethodCode: 'test',
            orderId: 1,
            baseEventTemplateName: 'test',
            parameterValues: { test: 'test' },
            type: 'event'
          }
        },
        rules: [{
          name: 'testRule',
          enabled: true,
          actions: testActions,
          feId: 'test',
          triggers: [],
          condition: {},
          orderId: 0
        }]
      }
    }
  };

  test('Successfully loads methods', () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);

    expect(moduleDescriptor.methods.length).toBe(2);
    expect(moduleDescriptor.methods[0].name).toBe('testMethod1');
    expect(moduleDescriptor.methods[0].factory).toBe(true);
    expect(moduleDescriptor.methods[1].name).toBe('testMethod2');
    expect(moduleDescriptor.methods[1].factory).toBe(false);
  });

  test('Successfully load events', () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);

    expect(moduleDescriptor.events.length).toBe(2);
    expect(moduleDescriptor.events[0].name).toBe('testEvent1');
    expect(moduleDescriptor.events[0].factory).toBe(true);
    expect(moduleDescriptor.events[1].name).toBe('testEvent2');
    expect(moduleDescriptor.events[1].factory).toBe(false);
  });

  test('Successfully load variables', () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);

    expect(moduleDescriptor.variables.length).toBe(2);
    expect(moduleDescriptor.variables[0].name).toBe('testVariable1');
    expect(moduleDescriptor.variables[0].factory).toBe(true);
    expect(moduleDescriptor.variables[1].name).toBe('testVariable2');
    expect(moduleDescriptor.variables[1].factory).toBe(false);
  });

  test('Successfully load rules', () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);

    expect(moduleDescriptor.rules.length).toBe(1);
  });

  test('Successfully saves JSON descriptor', async () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);
    moduleDescriptor.saveDescriptor('', 'testModule', 'Test');

    expect(jest.mocked(fs).writeFile).toHaveBeenCalledWith('', JSON.stringify(moduleDescriptorJson, undefined, 2));
  });

  test('Successfully seperate children and extensions', async () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);
    const descriptors = [
      new MethodDescriptor(
        'testMethod1',
        [testParameter],
        new ActionsDescriptor(),
        '',
        'any',
        true,
        0
      ),
      new MethodDescriptor(
        'testMethod2',
        [testParameter],
        new ActionsDescriptor(),
        '',
        'any',
        false,
        0
      )
    ];

    const [children, extensions] = moduleDescriptor['seperateChildrenAndExtensions'](descriptors);

    expect(children[0].name).toBe('testMethod1');
    expect(extensions[0].name).toBe('testMethod2');
  });

  test('Successfully create module descriptor object', () => {
    const moduleDescriptor = new ModuleDescriptor();
    moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJson);

    const result = moduleDescriptor['createModuleDescriptorObject']({
      children: moduleDescriptorJson.Devices.Test.children,
      extensions: moduleDescriptorJson.Devices.Test.extensions,
      moduleName: moduleDescriptorJson.Devices.Test.name,
      moduleId: 'Test'
    });

    expect(result).toEqual(moduleDescriptorJson);
  });
});
