"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Trigger_1 = require("./Trigger");
const cron_1 = require("cron");
var JobType;
(function (JobType) {
    JobType["PeriodicWoStartWoEnd"] = "PeriodicWoStartWoEnd";
    JobType["PeriodicWithStartWithEnd"] = "PeriodicWithStartWithEnd";
    JobType["PeriodicWoStartWithEnd"] = "PeriodicWoStartWithEnd";
    JobType["PeriodicWithStartWoEnd"] = "PeriodicWithStartWoEnd";
    JobType["RunsOnce"] = "RunsOnce";
    JobType["Invalid"] = "Invalid";
})(JobType || (JobType = {}));
class TimeTrigger extends Trigger_1.Trigger {
    constructor(options) {
        super(Trigger_1.TriggerType.Time, 0);
        this.cronJob = new cron_1.CronJob('* * * * * *', () => {
            super.start();
        });
        const jobType = this.getJobType(options);
        switch (jobType.jobType) {
            case JobType.PeriodicWithStartWithEnd: {
                const startDate = new Date(options.startDate);
                const endDate = new Date(options.endDate);
                if (!this.isExpired(startDate, endDate)) {
                    this.scheduleCronJobStart(startDate, options.cronTime);
                    this.scheduleCronJobStop(endDate);
                }
                break;
            }
            case JobType.PeriodicWithStartWoEnd: {
                const startDate = new Date(options.startDate);
                this.scheduleCronJobStart(startDate, options.cronTime);
                break;
            }
            case JobType.PeriodicWoStartWithEnd: {
                const startDate = new Date();
                const endDate = new Date(options.endDate);
                this.scheduleCronJobStart(startDate, options.cronTime);
                this.scheduleCronJobStop(endDate);
                break;
            }
            case JobType.PeriodicWoStartWoEnd: {
                this.startCronJob(options.cronTime);
                break;
            }
            case JobType.RunsOnce: {
                const startDate = new Date(options.startDate);
                const endDate = new Date(options.endDate);
                if (startDate.getTime() === endDate.getTime()) {
                    if (!this.isExpired(startDate, endDate)) {
                        this.scheduleOnce(startDate);
                    }
                }
                break;
            }
            default:
                console.warn(`Error during initializtaion of the TimeTrigger! ${jobType.message}`);
                break;
        }
    }
    getJobType(options) {
        if (options.cronTime !== undefined) {
            const result = this.isValidCrontime(options.cronTime);
            if (result.result === false) {
                return { jobType: JobType.Invalid, message: result.message };
            }
        }
        if (options.startDate && options.endDate) {
            const startDate = new Date(options.startDate);
            const endDate = new Date(options.endDate);
            if (startDate.getTime() > endDate.getTime()) {
                return { jobType: JobType.Invalid, message: `Start date ${startDate.toISOString()} is after the end date ${endDate.toISOString()} in time.` };
            }
            if (!options.cronTime) {
                if (startDate.getTime() === endDate.getTime()) {
                    return { jobType: JobType.RunsOnce }; // Runs exactly once on startDate.
                }
            }
            else {
                if (startDate.getTime() === endDate.getTime()) {
                    return { jobType: JobType.RunsOnce }; // Runs exactly once on startDate.
                }
                else {
                    return { jobType: JobType.PeriodicWithStartWithEnd }; // Periodic run with start date and end date.
                }
            }
        }
        else if (options.cronTime && !options.startDate && !options.endDate) { // Runs immediately. Theres is no end date. It runs infinitely
            return { jobType: JobType.PeriodicWoStartWoEnd };
        }
        else if (options.cronTime && options.startDate && !options.endDate) { // Starts at specfic time. Runs infinitely. Period is defined by crontime
            return { jobType: JobType.PeriodicWithStartWoEnd };
        }
        else if (options.cronTime && !options.startDate && options.endDate) { // Starts at current time. Runs until a specific time. Period is defined by crontime
            return { jobType: JobType.PeriodicWoStartWithEnd };
        }
        return { jobType: JobType.Invalid, message: 'Options of the TimerTrigger are invalid!' };
    }
    isValidCrontime(cronTime) {
        try {
            new cron_1.CronTime(cronTime);
            return { result: true };
        }
        catch (e) {
            return {
                result: false,
                message: `Error occured during initializtaion of the cron job! Cron time has a invalid format: ${e.message}.`
            };
        }
    }
    startCronJob(cronTime) {
        this.cronJob.setTime(new cron_1.CronTime(cronTime));
        this.cronJob.start();
    }
    stopCronJob() {
        this.cronJob.stop();
        super.reset();
    }
    scheduleCronJobStart(startDate, cronTime) {
        const startTimeOut = startDate.getTime() - Date.now();
        setTimeout(() => {
            this.startCronJob(cronTime);
        }, startTimeOut);
    }
    scheduleCronJobStop(endDate) {
        const endTimeOut = endDate.getTime() - Date.now();
        setTimeout(() => {
            this.stopCronJob();
        }, endTimeOut);
    }
    scheduleOnce(startDate) {
        setTimeout(() => {
            super.start();
        }, startDate.getTime() - Date.now());
    }
    isExpired(startDate, endDate) {
        if ((startDate.getTime() >= Date.now() && endDate.getTime() >= Date.now()) || (endDate.getTime() >= Date.now())) {
            return false;
        }
        return true;
    }
}
exports.default = TimeTrigger;
//# sourceMappingURL=TimeTrigger.js.map