"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LocalVariableManager = void 0;
const IpcMessage_1 = require("./Communication/IpcMessage");
const MessageParser_1 = require("./Communication/MessageParser");
const VariableAddedMessage_1 = require("./Communication/VariableAddedMessage");
const VariableChangedMessage_1 = require("./Communication/VariableChangedMessage");
const VariableGetRemoteResponse_1 = require("./Communication/VariableGetRemoteResponse");
const VariableMessageType_1 = require("./Communication/VariableMessageType");
const VariableStateMessageData_1 = require("./Communication/VariableStateMessageData");
const LocalVariable_1 = require("./LocalVariable");
class LocalVariableManager {
    /**
     * @param localInstanceid The instance id of the owner instance, eg.: taurus.
     * @param comm
     */
    constructor(localInstanceid, comm) {
        this.localInstanceid = localInstanceid;
        this.comm = comm;
        this.variables = {};
        this.proxy = new Proxy(this.variables, {
            get: (obj, prop) => {
                const variable = obj[prop];
                if (typeof (variable) == 'undefined') {
                    console.warn(`Variable (${prop.toString()}), not exists. Thus cannot be accessed through proxy object!`);
                    return undefined;
                }
                return Reflect.get(obj, prop);
            },
            set: (_obj, prop, _value) => {
                console.warn(`Variable (${prop.toString()}), can not be set through proxy. The form of ${prop.toString()}.value = <x> has to be used!`);
                return true;
            }
        });
    }
    createVariable(name, typeSignature, ownerId, defaultValue) {
        if (!this.variables[name]) {
            const variable = new LocalVariable_1.LocalVariable(name, typeSignature, ownerId, defaultValue);
            variable.onChangedInternal((variable) => {
                const messageData = VariableStateMessageData_1.VariableStateMessageData.create(variable);
                const messageId = (0, IpcMessage_1.generateUniqueId)(variable.ownerId);
                const message = new VariableChangedMessage_1.VariableChangedMessage(messageId, variable.ownerId, 'placeholder', messageData);
                this.sendMessage(message);
            });
            this.addVariable(variable);
        }
        return this.variables[name];
    }
    getVariable(name) {
        return this.variables[name];
    }
    receiveMessage(jsonMessage) {
        try {
            const message = (0, MessageParser_1.parseMessage)(jsonMessage);
            if (message !== null) {
                switch (message.messageType) {
                    case VariableMessageType_1.VariableMessageType.GetRemote: {
                        const variable = this.variables[message.messageData.variableName];
                        let messageData;
                        let response;
                        if (variable != undefined) {
                            messageData = VariableStateMessageData_1.VariableStateMessageData.create(variable);
                            response = new VariableGetRemoteResponse_1.VariableGetRemoteResponse(message.messageId, message.source, variable.ownerId, messageData);
                        }
                        else {
                            messageData = VariableStateMessageData_1.VariableStateMessageData.createUndefined(message.messageData.variableName, this.localInstanceid);
                            response = new VariableGetRemoteResponse_1.VariableGetRemoteResponse(message.messageId, message.source, this.localInstanceid, messageData);
                        }
                        this.sendMessage(response);
                        break;
                    }
                }
            }
        }
        catch (error) {
            console.warn(error.message);
        }
    }
    sendMessage(message) {
        this.comm.send(message);
    }
    addVariable(variable) {
        if (!this.variables[variable.name]) {
            this.variables[variable.name] = variable;
            this.sendVariableAdded(this.variables[variable.name]);
        }
    }
    sendVariableAdded(variable) {
        const messageData = VariableStateMessageData_1.VariableStateMessageData.create(variable);
        const messageId = (0, IpcMessage_1.generateUniqueId)(variable.ownerId);
        const message = new VariableAddedMessage_1.VariableAddedMessage(messageId, variable.ownerId, 'placeholder', messageData);
        this.sendMessage(message);
    }
}
exports.LocalVariableManager = LocalVariableManager;
//# sourceMappingURL=LocalVariableManager.js.map