"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoteVariable = void 0;
const events_1 = require("events");
const VariableChangedMessage_1 = require("./Communication/VariableChangedMessage");
const VariableGetRemoteResponse_1 = require("./Communication/VariableGetRemoteResponse");
const Variable_1 = require("./Variable");
const VariableAddedMessage_1 = require("./Communication/VariableAddedMessage");
const EventNames_1 = require("./EventNames");
class RemoteVariable {
    constructor(name, _ownerId) {
        this.name = name;
        this._ownerId = _ownerId;
        this.typeSignature = Variable_1.TypeSignature.Undefined;
        this._currentValue = undefined;
        this._previousValue = undefined;
        this._lastModifiedMs = -1;
        this.eventEmitter = new events_1.EventEmitter();
    }
    isValid() {
        return this._currentValue !== undefined;
    }
    get ownerId() {
        return this._ownerId;
    }
    get value() {
        return this._currentValue;
    }
    set value(_value) {
        const warnMessage1 = `The value of LARA variable ${this.name} (owned by instance ${this._ownerId}) can not be set .`;
        const warnMessage2 = `The value of a LARA variable can only be set by their owner.`;
        console.warn(`${warnMessage1} ${warnMessage2}`);
    }
    get previousValue() {
        return this._previousValue;
    }
    get lastModifiedMs() {
        return this._lastModifiedMs;
    }
    onValid(listener) {
        this.eventEmitter.on(EventNames_1.EventNames.Valid, listener);
    }
    onInvalid(listener) {
        this.eventEmitter.on(EventNames_1.EventNames.Invalid, listener);
    }
    onChanged(listener) {
        this.eventEmitter.on(EventNames_1.EventNames.Changed, listener);
    }
    onMessage(message) {
        if (message instanceof VariableAddedMessage_1.VariableAddedMessage ||
            message instanceof VariableChangedMessage_1.VariableChangedMessage ||
            message instanceof VariableGetRemoteResponse_1.VariableGetRemoteResponse) {
            this.setInternalState(message.messageData);
        }
    }
    setLastModifiedMs(time) {
        if (this._lastModifiedMs > time) {
            console.warn(`The incoming timestamp (${time} ms) is earlier then the current timestamp (${this._lastModifiedMs} ms) in the ${this._ownerId}.${this.name} status variable!`);
        }
        this._lastModifiedMs = time;
    }
    setInternalState(messageData) {
        if (this._currentValue != messageData.currentValue) {
            let stateChange = Variable_1.StateChange.None;
            if (this.isValid() === false && messageData.currentValue !== undefined) {
                stateChange = Variable_1.StateChange.ToValid;
            }
            else if (this.isValid() === true && messageData.currentValue === undefined) {
                stateChange = Variable_1.StateChange.ToInvalid;
            }
            this._currentValue = messageData.currentValue;
            this._previousValue = messageData.previousValue;
            this.typeSignature = messageData.typeSignature;
            this.setLastModifiedMs(messageData.lastModifiedMs);
            if (stateChange == Variable_1.StateChange.ToValid) {
                this.eventEmitter.emit(EventNames_1.EventNames.Valid);
            }
            else if (stateChange == Variable_1.StateChange.ToInvalid) {
                this.eventEmitter.emit(EventNames_1.EventNames.Invalid);
            }
            this.eventEmitter.emit(EventNames_1.EventNames.Changed, this._currentValue, this._previousValue);
        }
    }
}
exports.RemoteVariable = RemoteVariable;
//# sourceMappingURL=RemoteVariable.js.map