"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const LocalVariableManager_1 = require("../LocalVariableManager");
const LocalVariable_1 = require("../LocalVariable");
const Variable_1 = require("../Variable");
const VariableMessageType_1 = require("../Communication/VariableMessageType");
describe('LocalVariableManager', () => {
    afterEach(() => {
        jest.clearAllMocks();
    });
    const consoleWarnSpy = jest.spyOn(global.console, 'warn');
    const mockConnection = {
        send: jest.fn()
    };
    const localInstanceId = "ownerId";
    test('Successfull creation of an instance', () => {
        expect(() => {
            new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        }).not.toThrow();
        expect(new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection)).toBeInstanceOf(LocalVariableManager_1.LocalVariableManager);
    });
    test('Successful call of createVariable', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        expect(() => {
            lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.String, localInstanceId, 1);
        }).not.toThrow();
    });
    test('Successfull variable creation', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        const variable = lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.Number, localInstanceId, 1);
        expect(variable).toBeInstanceOf(LocalVariable_1.LocalVariable);
        expect(variable.name).toBe("testVariable");
        expect(variable.typeSignature).toBe(Variable_1.TypeSignature.Number);
        expect(variable.value).toBe(1);
    });
    test('Successful set of variable via variable manager', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        const variable = lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.Number, localInstanceId, 1);
        expect(() => {
            lvmInstance.getVariable("testVariable").value = 2;
        }).not.toThrow();
        expect(lvmInstance.getVariable("testVariable").value).toBe(2);
        expect(variable.value).toBe(2);
        expect(lvmInstance.proxy["testVariable"].value).toBe(2);
    });
    test('Fail set of non-existent variable via variable manager', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        expect(() => {
            lvmInstance.proxy["testVariable"].value = 1;
        }).toThrow(TypeError);
        expect(lvmInstance.getVariable("testVariable")).toBe(undefined);
        expect(consoleWarnSpy).toHaveBeenCalledWith("Variable (testVariable), not exists. Thus cannot be accessed through proxy object!");
    });
    test('Set variable via proxy', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        lvmInstance.proxy["cica2"] = 10;
        expect(consoleWarnSpy).toHaveBeenCalledWith("Variable (cica2), can not be set through proxy. The form of cica2.value = <x> has to be used!");
        expect(lvmInstance.proxy["cica2"]).toBe(undefined);
        expect(consoleWarnSpy).toHaveBeenNthCalledWith(1, "Variable (cica2), can not be set through proxy. The form of cica2.value = <x> has to be used!");
        expect(consoleWarnSpy).toHaveBeenNthCalledWith(2, "Variable (cica2), not exists. Thus cannot be accessed through proxy object!");
    });
    test('Fail get of non-existent variable via variable manager', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        const variable = lvmInstance.proxy["testVariable"];
        expect(variable).toBe(undefined);
        expect(consoleWarnSpy).toHaveBeenCalledWith("Variable (testVariable), not exists. Thus cannot be accessed through proxy object!");
    });
    test('Check previous value after set via variable manager', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        const variable = lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.Number, localInstanceId, 1);
        expect(() => {
            lvmInstance.getVariable("testVariable").value = 2;
        }).not.toThrow();
        expect(variable.previousValue).toBe(1);
    });
    test('Fail receiveMessage due to bad JSON string parameter', () => {
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        lvmInstance.receiveMessage("NOT a JSON string");
        expect(consoleWarnSpy).toHaveBeenCalledWith("Unexpected token N in JSON at position 0");
    });
    test('Successfully call receiveMessage with valid JSON string parameter - without sendMessage', () => {
        //Modify the LocalVariableManager proto to be able to mock and spy the sendMessage private function
        const sendMessageSpy = jest.spyOn(LocalVariableManager_1.LocalVariableManager.prototype, 'sendMessage').mockImplementation(jest.fn());
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.Number, localInstanceId, 1);
        //Type correct, fake messageObject
        const messageObject = {
            "messageType": VariableMessageType_1.VariableMessageType.Changed,
            "messageId": "1",
            "instanceId": "1",
            "source": "test",
            "messageData": {
                "variableName": "testVariable",
                "ownerId": localInstanceId,
                "currentValue": 1,
                "previousValue": 2,
                "lastModifiedMs": 1
            }
        };
        expect(() => {
            lvmInstance.receiveMessage(JSON.stringify(messageObject));
        }).not.toThrow();
        // Expected value is 1, because in createVariable function the sendMessage has been called, but in the receiveMessage the VariableMessageType is Changed so that's why in this case sendMessage has not been called.
        expect(sendMessageSpy).toHaveBeenCalledTimes(1);
        expect(consoleWarnSpy).not.toHaveBeenCalled();
    });
    test('Successfully call receiveMessage with valid JSON string parameter - with sendMessage, variable defined', () => {
        //Modify the LocalVariableManager proto to be able to mock and spy the sendMessage private function
        const sendMessageSpy = jest.spyOn(LocalVariableManager_1.LocalVariableManager.prototype, 'sendMessage').mockImplementation(jest.fn());
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        lvmInstance.createVariable("testVariable", Variable_1.TypeSignature.Number, localInstanceId, 1);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.GetRemote,
            messageData: { variableName: 'testVariable' }
        };
        expect(() => {
            lvmInstance.receiveMessage(JSON.stringify(jsonMessage));
        }).not.toThrow();
        expect(sendMessageSpy).toHaveBeenCalledTimes(2);
        expect(consoleWarnSpy).not.toHaveBeenCalled();
    });
    test('Successfully call receiveMessage with valid JSON string parameter - with sendMessage, variable not defined', () => {
        //Modify the LocalVariableManager proto to be able to mock and spy the sendMessage private function
        const sendMessageSpy = jest.spyOn(LocalVariableManager_1.LocalVariableManager.prototype, 'sendMessage').mockImplementation(jest.fn());
        const lvmInstance = new LocalVariableManager_1.LocalVariableManager(localInstanceId, mockConnection);
        // lvmInstance.createVariable("testVariable", TypeSignature.Number, localInstanceId, 1);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.GetRemote,
            messageData: { variableName: 'cica2' }
        };
        expect(() => {
            lvmInstance.receiveMessage(JSON.stringify(jsonMessage));
        }).not.toThrow();
        expect(sendMessageSpy).toHaveBeenCalledTimes(1);
        expect(consoleWarnSpy).not.toHaveBeenCalled();
    });
});
//# sourceMappingURL=LocalVariableManager.test.js.map