"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const RemoteVariableManager_1 = require("../RemoteVariableManager");
const RemoteVariable_1 = require("../RemoteVariable");
const VariableMessageType_1 = require("../Communication/VariableMessageType");
const Variable_1 = require("../Variable");
describe('RemoteVariableManager', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });
    const mockConnection = {
        send: jest.fn()
    };
    const consoleWarnSpy = jest.spyOn(console, 'warn').mockImplementationOnce(() => { });
    test('Successfully create a RemoteVariableManager', () => {
        expect(() => {
            new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        }).not.toThrow();
    });
    test('getVariable run without error', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        expect(() => {
            remoteVariableManager.getVariable("cica");
        }).not.toThrow();
    });
    test('create remote variable by getVariable', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        expect(remoteVariableManager.getVariable("cica2")).toBeInstanceOf(RemoteVariable_1.RemoteVariable);
    });
    test('getVariable on not exist variable log warn message', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        expect(() => {
            remoteVariableManager.proxy["cica123"];
        }).not.toThrow();
        expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
        expect(consoleWarnSpy).toHaveBeenCalledWith("LARA variable cica123 not exists on cica or not ready yet. If cica does have LARA variable cica123 then you probably need to wait for an appropriate event before accessing it.");
    });
    test('createPlaceholder run without error and check value', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        expect(() => {
            remoteVariableManager.createVariablePlaceholder("cica2");
        }).not.toThrow();
        expect(remoteVariableManager.proxy["cica2"].value).toBe(undefined);
    });
    test('Set variable via getVariable', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        expect(() => {
            remoteVariableManager.getVariable("cica").value = 10;
        }).not.toThrow();
        expect(remoteVariableManager.proxy["cica"].value).toBe(undefined);
    });
    test('Set variable via proxy', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica2", "mica", mockConnection);
        let returnValue = null;
        remoteVariableManager.proxy["cica2"] = 10;
        expect(consoleWarnSpy).toBeCalledWith("Variable cica2, owned by cica2, can not be set through proxy. The form of cica2.value = <x> has to be used!");
        expect(remoteVariableManager.proxy["cica2"].value).toBe(undefined);
        expect(consoleWarnSpy).toBeCalledWith("LARA variable cica2 not exists on cica2 or not ready yet. If cica2 does have LARA variable cica2 then you probably need to wait for an appropriate event before accessing it.");
        expect(returnValue).toBe(null);
    });
    test('receiveMessage with wrong jsonMessage', () => {
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("cica", "mica", mockConnection);
        remoteVariableManager.getVariable("cica2");
        remoteVariableManager.getVariable("cica3");
        expect(() => {
            remoteVariableManager.receiveMessage("cica123");
        }).not.toThrow();
        expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
        expect(consoleWarnSpy).toHaveBeenLastCalledWith("Unexpected token c in JSON at position 0");
    });
    test('receiveMessage with right jsonMessage', () => {
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica2',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123,
                ownerId: 'taurus',
                typeSignature: Variable_1.TypeSignature.Number,
            }
        };
        const remoteVariableManager = new RemoteVariableManager_1.RemoteVariableManager("taurus", "mica", mockConnection);
        remoteVariableManager.getVariable("cica2");
        expect(() => {
            remoteVariableManager.receiveMessage(JSON.stringify(jsonMessage));
        }).not.toThrow();
        expect(remoteVariableManager.proxy["cica2"].value).toBe(10);
        expect(remoteVariableManager.getVariable("cica2").value).toBe(10);
        expect(remoteVariableManager.getVariable("cica2").previousValue).toBe(20);
        expect(remoteVariableManager.getVariable("cica2").lastModifiedMs).toBe(123);
    });
});
//# sourceMappingURL=RemoteVariableManager.test.js.map