import { writeFile } from "fs/promises";
import { ActionsDescriptor } from "./ActionsDescriptor";
import { EventDescriptor } from "./EventDescriptor";
import { MethodCodeData, MethodDescriptor } from "./MethodDescriptor";
import { Descriptors, Device, ElementType, ModuleDescriptorObject, Parameter } from "./ModuleDescriptorObject";
import { RuleDescriptor, LegacyRuleDescriptor } from "./RuleDescriptor";
import { VariableDescriptor } from "./VariableDescriptor";
import { readFileSync } from "fs";
import assert from 'node:assert/strict';
import { TriggerType } from "./TriggerDescriptor";

export class ModuleDescriptorData {
  public methods: Array<MethodDescriptor> = [];
  public events: Array<EventDescriptor> = [];
  public rules: Array<RuleDescriptor> = [];
  public variables: Array<VariableDescriptor> = [];
}

export class ModuleDescriptor extends ModuleDescriptorData {
  private methodsFromEntryCodeData: MethodCodeData = {};
  constructor() {
    super();
  }

  public readDescriptor(descriptorPath: string, entryCodePath?: string): void {
    const object = JSON.parse(readFileSync(descriptorPath, { encoding: 'utf8' }));
    if (entryCodePath) {
      const entryCode = readFileSync(entryCodePath, { encoding: 'utf8' });
      return this.loadFromDescriptorObject(object, entryCode);
    }
    return this.loadFromDescriptorObject(object);
  }

  public loadFromDescriptorObject(moduleDescriptorObject: ModuleDescriptorObject, moduleEntryCode?: string): void {
    if (moduleEntryCode) {
      this.methodsFromEntryCodeData = MethodDescriptor.getMethodsFromEntryCode(moduleEntryCode);
    }
    for (const device in moduleDescriptorObject.Devices) {
      if (typeof moduleDescriptorObject.Devices[device].rules == 'undefined') {
        moduleDescriptorObject.Devices[device].rules = [];
      }

      if (typeof moduleDescriptorObject.Devices[device].extensions == 'undefined') {
        moduleDescriptorObject.Devices[device].extensions = {};
      }

      this.loadMethods(moduleDescriptorObject.Devices[device], moduleEntryCode);
      this.loadVariables(moduleDescriptorObject.Devices[device]);
      this.loadEvents(moduleDescriptorObject.Devices[device]);
      this.loadRules(moduleDescriptorObject.Devices[device]);
    }
  }

  public loadFromModuleDescriptorData(moduleDescriptorData: ModuleDescriptorData): void {
    this.methods = moduleDescriptorData.methods;
    this.events = moduleDescriptorData.events;
    this.rules = moduleDescriptorData.rules;
    this.variables = moduleDescriptorData.variables;
  }

  private loadMethods(device: Device, moduleEntryCode?: string): void {
    const load = (data: Record<string, any>, factory: boolean): void => {
      for (const [key, child] of Object.entries(data)) {
        if (child.type === ElementType.Method) {
          this.methods.push(this.initMethod(key, child, factory, moduleEntryCode));
        }
      }
    };

    load(device.children, true);
    load(device.extensions, false);
  }

  private initMethod(key: string, child: any, factory: boolean, moduleEntryCode?: string): MethodDescriptor {
    const actions = new ActionsDescriptor(
      moduleEntryCode ? this.methodsFromEntryCodeData[key].code : '',
      child.actions?.type,
      child.actions?.steps
    );
    return new MethodDescriptor(
      key,
      child.parameters,
      actions,
      moduleEntryCode ? this.methodsFromEntryCodeData[key].description : child.description,
      moduleEntryCode ? this.methodsFromEntryCodeData[key].returnType : child.returnType,
      factory,
      child.orderId
    );
  }

  private loadEvents(device: Device): void {
    const load = (data: Record<string, any>, factory: boolean): void => {
      for (const [key, child] of Object.entries(data)) {
        if (child.type === ElementType.Event) {
          this.events.push(this.initEvents(key, child, factory));
        }
      }
    };

    load(device.children, true);
    load(device.extensions, false);
  }

  private initEvents(key: string, child: any, factory: boolean): EventDescriptor {
    const actions = new ActionsDescriptor(child.actions?.code, child.actions?.type, child.actions?.steps);
    return new EventDescriptor(
      key,
      child.callbackParameters,
      actions,
      child.description,
      factory,
      child.orderId,
      child.baseEventTemplateName,
      child.parameterValues
    );
  }

  private loadVariables(device: Device): void {
    const load = (data: Record<string, any>, factory: boolean): void => {
      for (const [key, child] of Object.entries(data)) {
        if (child.type === ElementType.Variable) {
          const variableDescriptor: VariableDescriptor = new VariableDescriptor(
            key,
            child.valueType,
            child.defaultValue,
            child.description,
            child.orderId,
            factory
          );

          this.variables.push(variableDescriptor);
        }
      }
    };

    load(device.children, true);
    load(device.extensions, false);
  }

  private loadRules(device: Device): void {
    const convertedRules = this.convertRules(device.rules);
    convertedRules.map((rule) => {
      const actions = new ActionsDescriptor(rule.actions?.code, rule.actions?.type, rule.actions?.steps);
      this.rules.push(new RuleDescriptor(
        rule.name,
        rule.enabled!,
        actions,
        rule.feId,
        rule.triggers!,
        rule.condition,
        rule.factory,
        rule.orderId
      ));
    });
  }

  /**
   * @brief Compares the interface of the modules. The User additions are ignored.
   *        Method signatures and event callback signatures are compared.
   *        The method and event codes are ignored.
   * @param other The other ModuleDescrirptor to compare with.
   * @returns true if the Moduledescriptors are compatible with each other.
   */
  public isCompatibleWith(other: ModuleDescriptor): boolean {
    for (const configMethod of this.methods) {
      if (!configMethod.factory) continue;

      const factoryMehtod = other.methods.find((factoryMethod) => factoryMethod.name === configMethod.name);

      if (factoryMehtod === undefined || !this.compareMethodSignature(factoryMehtod, configMethod)) {
        return false;
      }
    }

    for (const configEvent of this.events) {
      if (!configEvent.factory) continue;

      const factoryEvent = other.events.find((factoryEvent) => factoryEvent.name === configEvent.name);

      if (
        factoryEvent === undefined ||
        !this.compareParameters(factoryEvent.callbackParameters, configEvent.callbackParameters)
      ) {
        return false;
      }
    }

    for (const configVariable of this.variables) {
      if (!configVariable.factory) continue;

      const factoryVariable = other.variables.find((factoryVariable) => factoryVariable.name === configVariable.name);

      let deepEqual = true;
      try {
        assert.deepStrictEqual(factoryVariable, configVariable);
      } catch (e) {
        deepEqual = false;
      }

      if (factoryVariable === undefined || !deepEqual) {
        return false;
      }
    }

    return true;
  }

  private compareMethodSignature(originalMethod: MethodDescriptor, newMethod: MethodDescriptor): boolean {
    if (newMethod.returnType != originalMethod.returnType) return false;

    if (!this.compareParameters(originalMethod.parameters, newMethod.parameters)) return false;

    return true;
  }

  private compareParameters(params1: Parameter[], params2: Parameter[]): boolean {
    if (
      (typeof params1 === 'undefined' && typeof params2 !== 'undefined') ||
      (typeof params1 !== 'undefined' && typeof params2 === 'undefined')
    ) {
      return false;
    } else if (typeof params1 === 'undefined' && typeof params2 === 'undefined') {
      return true;
    }

    // Check parameter count
    if (params1.length != params2.length) return false;

    // Check parameter names, types
    for (let i = 0; i < params2.length; i++) {
      if (params2[i].name != params1[i].name) return false;
      if (params2[i].type != params1[i].type) return false;
    }

    return true;
  }

  /**
   * @brief   Gets the extension descriptor of the module, basically returns all the elements
   *          which factory property equals to false.
   * @returns ModuleDescriptorData with the user additive elments.
   */
  public getExtensions(): ModuleDescriptorData {
    const data = new ModuleDescriptorData();

    for (const method of this.methods) {
      if (method.factory == false) {
        data.methods.push(method);
      }
    }

    for (const event of this.events) {
      if (event.factory == false) {
        data.events.push(event);
      }
    }

    for (const variable of this.variables) {
      if (variable.factory == false) {
        data.variables.push(variable);
      }
    }

    data.rules = this.rules;

    return data;
  }

  /**
   * @brief   Check if the rules in the module are in the old format (event instead of trigger)
   * @returns void
   */
  public convertRules(rules: LegacyRuleDescriptor[]): LegacyRuleDescriptor[] {
    if (rules.length > 0) {
      rules.map((rule: LegacyRuleDescriptor) => {
        if (rule.event && !rule.triggers) {
          rule.triggers = [{
            type: TriggerType.Event,
            details: {
              instanceId: rule.event.instanceId,
              event: rule.event.event
            }
          }];

          delete rule.event;
        }

        if (typeof rule.enabled == "undefined") {
          rule.enabled = true;
        }
      });
    }

    return rules;
  }

  public async saveDescriptor(path: string, moduleName: string, moduleId: string): Promise<void> {
    const [childrenMethods, extensionMethods] = this.seperateChildrenAndExtensions<MethodDescriptor>(this.methods);
    const [childrenVariables, extensionVariables] = this.seperateChildrenAndExtensions<VariableDescriptor>(this.variables);
    const [childrenEvents, extensionEvents] = this.seperateChildrenAndExtensions<EventDescriptor>(this.events);

    const children: Record<string, any> = {};
    childrenMethods.map((descriptor) => {
      children[descriptor.name] = {
        ...descriptor,
        name: undefined,
        factory: undefined,
        type: 'method',
        actions: { type: descriptor.actions.type, steps: descriptor.actions.steps, code: '' }
      };

    });
    childrenVariables.map((descriptor) => {
      children[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'variable' };
    });
    childrenEvents.map((descriptor) => {
      children[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'event' };
    });

    const extensions: Record<string, any> = {};
    extensionMethods.map((descriptor) => {
      extensions[descriptor.name] = {
        ...descriptor,
        name: undefined,
        factory: undefined,
        type: 'method',
        actions: { type: descriptor.actions.type, steps: descriptor.actions.steps, code: '' }
      };

    });
    extensionVariables.map((descriptor) => {
      extensions[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'variable' };
    });
    extensionEvents.map((descriptor) => {
      extensions[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'event' };
    });


    const moduleDescriptorObject = this.createModuleDescriptorObject({ children, extensions, moduleName, moduleId });

    await writeFile(path, JSON.stringify(moduleDescriptorObject, undefined, 2));
  }

  private seperateChildrenAndExtensions<T extends Descriptors>(
    descriptors: Array<T>
  ): [Array<T>, Array<T>] {
    const children: Array<T> = [];
    const extensions: Array<T> = [];

    descriptors.map((descriptor) => {
      if (descriptor.factory) {
        children.push(descriptor);
      } else {
        extensions.push(descriptor);
      }
    });

    return [children, extensions];
  }

  private createModuleDescriptorObject({
    children,
    extensions,
    moduleName,
    moduleId
  }: {
    children: Record<string, any>,
    extensions: Record<string, any>,
    moduleName: string,
    moduleId: string
  }): ModuleDescriptorObject {
    const rules = this.rules.map((r) => ({ ...r, factory: undefined, type: undefined }));
    return {
      Schemas: {},
      Devices: {
        [moduleId]: {
          name: moduleName,
          type: 'group',
          children,
          extensions,
          rules
        }
      }
    };
  }
}
