"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const TimeTrigger_1 = __importDefault(require("./TimeTrigger"));
jest.useFakeTimers();
const consoleWarnSpy = jest.spyOn(global.console, 'warn');
describe('Time based trigger test', () => {
    describe('TimeTrigger JobTypes', () => {
        test('PeriodicWoStartWoEnd', () => {
            const trigger = new TimeTrigger_1.default({ cronTime: '* * * * * *' });
            trigger['stopCronJob']();
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('PeriodicWithStartWithEnd', () => {
            new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: '2100-01-01 02:00:00', endDate: '2100-01-01 03:00:00' });
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('PeriodicWoStartWithEnd', () => {
            new TimeTrigger_1.default({ cronTime: '* * * * * *', endDate: '2100-01-01 03:00:00' });
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('PeriodicWithStartWoEnd', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const trigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: '2100-01-01 01:00:00' });
            setTimeout(() => {
                trigger['stopCronJob']();
            }, 1000);
            const timeAdvance = new Date('2100-01-01 01:02:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('RunsOnce', () => {
            new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: '2100-01-01 02:00:00', endDate: '2100-01-01 02:00:00' });
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('RunsOnce', () => {
            new TimeTrigger_1.default({ startDate: '2100-01-01 02:00:00', endDate: '2100-01-01 02:00:00' });
            expect(consoleWarnSpy).not.toBeCalled();
        });
        test('Invalid JobType', () => {
            new TimeTrigger_1.default({});
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Options of the TimerTrigger are invalid!");
        });
    });
    describe('StartDate and EndDate are Date objects', () => {
        test('Create time trigger where the endDate is earlier than the srartDate. Warning is expected.', () => {
            const options = {
                cronTime: '* * * * * *',
                startDate: new Date('2100-01-01 02:00:00'),
                endDate: new Date('2100-01-01 01:00:00')
            };
            new TimeTrigger_1.default(options);
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Start date 2100-01-01T01:00:00.000Z is after the end date 2100-01-01T00:00:00.000Z in time.");
        });
        test('Create time trigger with invalid cron format.', () => {
            const startDate = new Date('2100-01-01 02:00:00');
            const endDate = new Date('2100-01-01 03:00:00');
            new TimeTrigger_1.default({ cronTime: '', startDate: startDate, endDate: endDate });
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Error occured during initializtaion of the cron job! Cron time has a invalid format: Too few fields.");
            new TimeTrigger_1.default({ cronTime: '* * * * * asd', startDate: startDate, endDate: endDate });
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Error occured during initializtaion of the cron job! Cron time has a invalid format: Unknown alias: asd.");
        });
        test('Create time trigger which runs once at a specific date.', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const startDate = new Date('2100-01-01 01:01:00');
            const endDate = new Date('2100-01-01 01:01:00');
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 01:02:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(1);
        });
        test('Create time trigger which runs periocically every second', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const startDate = new Date('2100-01-01 02:00:00');
            const endDate = new Date('2100-01-01 02:00:10');
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 03:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(9);
        });
        test('Create time trigger which runs periocically every day at 8:00:00', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '0 0 8 * * *' });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const times = [];
            timeTrigger.onTriggered(() => {
                times.push(new Date().toISOString());
            });
            const timeAdvance = new Date('2100-01-05 23:59:59').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(5);
            const expectedOutput = [
                '2100-01-01T07:00:00.000Z',
                '2100-01-02T07:00:00.000Z',
                '2100-01-03T07:00:00.000Z',
                '2100-01-04T07:00:00.000Z',
                '2100-01-05T07:00:00.000Z'
            ];
            expect(times).toEqual(expectedOutput);
        });
    });
    describe('StartDate and EndDate are strings', () => {
        test('Create time trigger where the endDate is earlier than the srartDate. Warning is expected.', () => {
            new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: '2100-01-01 02:00:00', endDate: '2100-01-01 01:00:00' });
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Start date 2100-01-01T01:00:00.000Z is after the end date 2100-01-01T00:00:00.000Z in time.");
        });
        test('Create time trigger where the start date is before the Date.now(), and the endDate is after Date.now()', () => {
            jest.setSystemTime(new Date('2100-01-01 02:00:05'));
            const startDate = '2100-01-01 02:00:00';
            const endDate = '2100-01-01 02:00:10';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 03:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(4);
        });
        test('Create time trigger where the start date is before the Date.now(), and the endDate is before Date.now()', () => {
            jest.setSystemTime(new Date('2100-01-01 02:00:11'));
            const startDate = '2100-01-01 02:00:00';
            const endDate = '2100-01-01 02:00:10';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 03:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(0);
        });
        test('Create time trigger where the start date is before the Date.now(), and the endDate is before Date.now()', () => {
            jest.setSystemTime(new Date('2100-01-01 02:00:08'));
            const startDate = '2100-01-01 02:00:00';
            const endDate = '2100-01-01 02:00:10';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 03:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(1);
        });
        test('Create time trigger with invalid cron format.', () => {
            const startDate = '2100-01-01 02:00:00';
            const endDate = '2100-01-01 01:00:00';
            new TimeTrigger_1.default({ cronTime: '', startDate: startDate, endDate: endDate });
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Error occured during initializtaion of the cron job! Cron time has a invalid format: Too few fields.");
            new TimeTrigger_1.default({ cronTime: '* * * * * asd', startDate: startDate, endDate: endDate });
            expect(consoleWarnSpy).toHaveBeenCalledWith("Error during initializtaion of the TimeTrigger! Error occured during initializtaion of the cron job! Cron time has a invalid format: Unknown alias: asd.");
        });
        test('Create time trigger which runs once at a specific date.', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const startDate = '2100-01-01 01:01:00';
            const endDate = '2100-01-01 01:01:00';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 01:02:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(1);
        });
        test('Create time trigger which should run at once on a specific date, but it is expired.', () => {
            jest.setSystemTime(new Date('2100-01-01 01:02:00'));
            const startDate = '2100-01-01 01:01:00';
            const endDate = '2100-01-01 01:01:00';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 01:02:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(0);
        });
        test('Create time trigger which runs periocically every second', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const startDate = '2100-01-01 02:00:00';
            const endDate = '2100-01-01 02:00:10';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate, endDate: endDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 03:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(9);
        });
        test('Create time trigger which runs periocically every day at 8:00:00', () => {
            jest.setSystemTime(new Date('2100-01-01 01:00:00'));
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '0 0 8 * * *' });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const times = [];
            timeTrigger.onTriggered(() => {
                times.push(new Date().toISOString());
            });
            const timeAdvance = new Date('2100-01-05 23:59:59').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(5);
            const expectedOutput = [
                '2100-01-01T07:00:00.000Z',
                '2100-01-02T07:00:00.000Z',
                '2100-01-03T07:00:00.000Z',
                '2100-01-04T07:00:00.000Z',
                '2100-01-05T07:00:00.000Z'
            ];
            expect(times).toEqual(expectedOutput);
        });
        test('Create time trigger where the start date is defined and the end date is undefined.', () => {
            jest.setSystemTime(new Date('2100-01-01 02:00:00'));
            const startDate = '2100-01-01 02:00:00';
            const timeTrigger = new TimeTrigger_1.default({ cronTime: '* * * * * *', startDate: startDate });
            const triggeredHandler = jest.fn();
            timeTrigger.onTriggered(triggeredHandler);
            const timeAdvance = new Date('2100-01-01 23:00:00').getTime() - Date.now();
            jest.advanceTimersByTime(timeAdvance);
            expect(triggeredHandler).toHaveBeenCalledTimes(75600);
        });
    });
});
//# sourceMappingURL=TimeTrigger.test.js.map