import {
  ActionsDescriptor,
  ConditionDescriptor,
  ConditionGroupDescriptor,
  ConditionGroupOperator,
  ConstOperandDescriptor,
  EventDescriptor,
  EventParameterOperandDescriptor,
  OperandType,
  Operator,
  VariableOperandDescriptor,
  EventTriggerDescriptor,
  TriggerType,
  VariableChangedToValueTriggerDescriptor,
  RuleDescriptor,
  TriggerDescriptor
} from "@lightware/module-descriptor";
import { RuleManager } from "./RuleManager";
import {
  Condition,
  ConditionGroup,
  ConstOperand,
  EventParameterOperand,
  VariableOperand
} from "@lightware/condition-api";
import { EventTrigger } from "../Triggers/EventTrigger";
import { VariableChangedToValueTrigger } from "../Triggers/VariableTrigger";
import { Rule } from "./Rule";
import TimeTrigger from "../Triggers/TimeTrigger";

jest.mock('@lightware/condition-api');
jest.useFakeTimers();

const mockGetInstanceById = jest.fn(() => ({
  on: jest.fn(),
  variables: { testVar: { onChanged: jest.fn() } }
}));
const mockgetModuleData = jest.fn();
const mockInstanceApi = {
  getModuleData: mockgetModuleData,
  getInstanceById: mockGetInstanceById,
  instanceId: 'testInstance'
};

describe('RuleManager', () => {
  let ruleManager: RuleManager;
  test('Successfully create RuleManager with empty module descriptor', () => {
    mockgetModuleData.mockReturnValueOnce({
      Devices: {
        Test: {
          name: 'test',
          type: 'group',
          children: {},
          rules: [],
          extensions: {}
        }
      }
    });

    expect(() => {
      ruleManager = new RuleManager(mockInstanceApi);
    }).not.toThrow();
  });

  describe('.getOperand', () => {
    test('Successfully return EventParameterOperand if operand type is eventCallbackParameter', () => {
      const operandDescriptor: EventParameterOperandDescriptor = {
        type: OperandType.EventCallbackParameter,
        instanceId: 'testInstanceId',
        name: 'test',
        event: 'testEvent'
      };
      const events = {
        [operandDescriptor.instanceId]: [new EventDescriptor(
          'testEvent',
          [{
            name: 'test',
            title: 'test',
            type: 'string'
          }],
          new ActionsDescriptor(),
          ''
        )]
      };

      const operand = ruleManager['getOperand'](operandDescriptor, events);

      expect(operand instanceof EventParameterOperand).toBe(true);
    });

    test('Throw an error if event with the name from the descriptor doesnt exist', () => {
      const operandDescriptor: EventParameterOperandDescriptor = {
        type: OperandType.EventCallbackParameter,
        instanceId: 'testInstanceId',
        name: 'test',
        event: 'testEvent1'
      };
      const events = {
        [operandDescriptor.instanceId]: [new EventDescriptor(
          'testEvent',
          [{
            name: 'test',
            title: 'test',
            type: 'string'
          }],
          new ActionsDescriptor(),
          ''
        )]
      };

      expect(() => {
        ruleManager['getOperand'](operandDescriptor, events);
      }).toThrow('Could not find event with name testEvent1');
    });

    test('Successfully return VariableOperand if operand type is StatusVariable', () => {
      const operandDescriptor: VariableOperandDescriptor = {
        type: OperandType.StatusVariable,
        instanceId: 'testInstanceId',
        name: 'test'
      };

      const operand = ruleManager['getOperand'](operandDescriptor, {});

      expect(operand instanceof VariableOperand).toBe(true);
    });

    test('Successfully return ConstOperand if operand type is Constant', () => {
      const operandDescriptor: ConstOperandDescriptor = {
        type: OperandType.Constant,
        value: 'test'
      };

      const operand = ruleManager['getOperand'](operandDescriptor, {});

      expect(operand instanceof ConstOperand).toBe(true);
    });

    test('Throws an error if operand type is unsupported', () => {
      const operandDescriptor = { type: 'test' };

      expect(() => {
        ruleManager['getOperand'](operandDescriptor as any, {});
      }).toThrow('Operand type: test, not supported');
    });
  });

  describe('.getConditionGroup', () => {
    const conditionDescriptor: ConditionDescriptor = {
      type: 'condition',
      left: {
        type: OperandType.StatusVariable,
        instanceId: 'testInstanceId',
        name: 'test'
      },
      operator: Operator.Equals,
      right: {
        type: OperandType.Constant,
        value: 'test'
      },
      duration: 0
    };
    test('Successfully return Condition instance if the type is condition', () => {
      const result = ruleManager['getConditionGroup'](conditionDescriptor, {});

      expect(result instanceof Condition).toBe(true);
    });

    test('Successfully returns ConditionGroup instance if the type is conditionGroup with 2 conditions in it', () => {
      const conditionGroupDescriptor: ConditionGroupDescriptor = {
        type: 'conditionGroup',
        operator: ConditionGroupOperator.AND,
        conditions: [conditionDescriptor, conditionDescriptor]
      };

      const result = ruleManager['getConditionGroup'](conditionGroupDescriptor, {});

      expect(result instanceof ConditionGroup).toBe(true);
    });

    test('Successfully returns ConditionGroup instance if the type is conditionGroup with 1 condition and 1 conditionGroup in it', () => {
      const innerConditionGroupDescriptor: ConditionGroupDescriptor = {
        type: 'conditionGroup',
        operator: ConditionGroupOperator.OR,
        conditions: [conditionDescriptor, conditionDescriptor]
      };
      const conditionGroupDescriptor: ConditionGroupDescriptor = {
        type: 'conditionGroup',
        operator: ConditionGroupOperator.AND,
        conditions: [innerConditionGroupDescriptor, conditionDescriptor]
      };

      const result = ruleManager['getConditionGroup'](conditionGroupDescriptor, {});

      expect(result instanceof ConditionGroup).toBe(true);
    });
  });

  describe('.getTriggers', () => {
    describe('TimeTriggers', () => {
      test('Successfully returns TimeTrigger instance', () => {
        jest.setTimeout(1000);


        const triggersDescriptor: TriggerDescriptor[] = [
          {
            type: TriggerType.Time,
            duration: 0,
            details: {
              cron: {
                seconds: '*',
                minutes: '*',
                hours: '*',
                dayOfMonth: '*',
                months: '*',
                dayOfWeek: '*'
              },
              startDate: '2023-05-24 09:27:01',
              endDate: '2024-05-24 09:27:01'
            }
          }
        ];

        const triggers = ruleManager['getTriggers'](triggersDescriptor, {});

        expect(triggers['triggers'].length).toBe(0);
        expect(triggers['triggers'][0] instanceof TimeTrigger).toBe(false);
      });
    });

    test('Successfully returns EventTrigger instance', () => {
      const triggersDescriptor: EventTriggerDescriptor[] = [
        {
          type: TriggerType.Event,
          details: {
            event: 'test',
            instanceId: 'testInstanceId'
          }
        }
      ];

      const triggers = ruleManager['getTriggers'](triggersDescriptor, {});

      expect(triggers['triggers'].length).toBe(1);
      expect(triggers['triggers'][0] instanceof EventTrigger).toBe(true);
    });

    test('Successfully returns VariableChangedToValueTrigger instance', () => {
      jest.setTimeout(1000);


      const triggersDescriptor: VariableChangedToValueTriggerDescriptor[] = [
        {
          type: TriggerType.VariableChangedToValue,
          duration: 10000,
          details: {
            instanceId: 'testInstance',
            variable: 'testVar',
            condition: {
              type: 'condition',
              left: {
                type: OperandType.StatusVariable,
                instanceId: 'testInstance',
                name: 'testVar'
              },
              operator: Operator.Equals,
              right: {
                type: OperandType.Constant,
                value: 50
              },
              duration: 0
            }
          }
        }
      ];

      const triggers = ruleManager['getTriggers'](triggersDescriptor, {});

      expect(triggers['triggers'].length).toBe(1);
      expect(triggers['triggers'][0] instanceof VariableChangedToValueTrigger).toBe(true);
    });
  });

  describe('.getRules', () => {
    test('Successfully returns a Rule with condition', () => {
      const ruleDescriptor = new RuleDescriptor(
        'ruleName',
        true,
        new ActionsDescriptor(),
        '',
        [],
        {
          type: 'condition',
          left: {
            type: OperandType.StatusVariable,
            instanceId: 'testInstance',
            name: 'testVar'
          },
          operator: Operator.Equals,
          right: {
            type: OperandType.Constant,
            value: 50
          },
          duration: 0
        }
      );

      const rules = ruleManager['getRules']([ruleDescriptor], {});
      ruleManager['rules'] = rules;

      expect(rules.length).toBe(1);
      expect(rules[0] instanceof Rule).toBe(true);
    });

    test('Successfully returns a Rule without condition', () => {
      const ruleDescriptor = new RuleDescriptor(
        'ruleName',
        true,
        new ActionsDescriptor(),
        '',
        []
      );

      const rules = ruleManager['getRules']([ruleDescriptor], {});
      ruleManager['rules'] = rules;

      expect(rules.length).toBe(1);
      expect(rules[0] instanceof Rule).toBe(true);
    });
  });

  describe('.setAction', () => {
    test('Successfully set action for an already created rule', () => {
      expect(() => {
        ruleManager.setAction('ruleName', async () => {
          console.log('test');
        });
      }).not.toThrow();
    });

    test('If rule can not be found call console.warn', () => {
      const warnSpy = jest.spyOn(console, 'warn').mockImplementation();
      ruleManager.setAction('test', async () => {
        console.log('test');
      });

      expect(warnSpy)
        .toHaveBeenCalledWith(
          'Rule with name test not found when trying to set an action for it in instance with id: testInstance'
        );
    });
  });
});
