import { Trigger, TriggerType } from "./Trigger";
import { CronJob, CronTime } from "cron";

export type Options = {
  cronTime?: string,
  startDate?: Date | string,
  endDate?: Date | string
}

enum JobType {
  PeriodicWoStartWoEnd = 'PeriodicWoStartWoEnd',
  PeriodicWithStartWithEnd = 'PeriodicWithStartWithEnd',
  PeriodicWoStartWithEnd = 'PeriodicWoStartWithEnd',
  PeriodicWithStartWoEnd = 'PeriodicWithStartWoEnd',
  RunsOnce = 'RunsOnce',
  Invalid = 'Invalid'
}

export default class TimeTrigger extends Trigger {
  private cronJob: CronJob = new CronJob('* * * * * *', () => {
    super.start();
  });

  constructor(options: Options) {
    super(TriggerType.Time, 0);

    const jobType = this.getJobType(options);

    switch (jobType.jobType) {
      case JobType.PeriodicWithStartWithEnd: {
        const startDate: Date = new Date(options.startDate!);
        const endDate: Date = new Date(options.endDate!);

        if (!this.isExpired(startDate, endDate)) {
          this.scheduleCronJobStart(startDate, options.cronTime!);
          this.scheduleCronJobStop(endDate);
        }

        break;
      }

      case JobType.PeriodicWithStartWoEnd: {
        const startDate: Date = new Date(options.startDate!);

        this.scheduleCronJobStart(startDate, options.cronTime!);

        break;
      }

      case JobType.PeriodicWoStartWithEnd: {
        const startDate: Date = new Date();
        const endDate: Date = new Date(options.endDate!);

        this.scheduleCronJobStart(startDate, options.cronTime!);
        this.scheduleCronJobStop(endDate);

        break;
      }

      case JobType.PeriodicWoStartWoEnd: {
        this.startCronJob(options.cronTime!);

        break;
      }

      case JobType.RunsOnce: {
        const startDate: Date = new Date(options.startDate!);
        const endDate: Date = new Date(options.endDate!);

        if (startDate.getTime() === endDate.getTime()) {
          if (!this.isExpired(startDate, endDate)) {
            this.scheduleOnce(startDate);
          }
        }

        break;
      }

      default:
        console.warn(`Error during initializtaion of the TimeTrigger! ${jobType.message}`);
        break;
    }
  }

  private getJobType(options: Options): {jobType: JobType, message?: string} {
    if (options.cronTime !== undefined) {
      const result = this.isValidCrontime(options.cronTime!);
      if (result.result === false) {
        return { jobType: JobType.Invalid, message: result.message };
      }
    }

    if (options.startDate && options.endDate) {
      const startDate: Date = new Date(options.startDate);
      const endDate: Date = new Date(options.endDate);

      if (startDate.getTime() > endDate.getTime()) {
        return { jobType: JobType.Invalid, message: `Start date ${startDate.toISOString()} is after the end date ${endDate.toISOString()} in time.` };
      }

      if (!options.cronTime) {
        if (startDate.getTime() === endDate.getTime()) {
          return { jobType: JobType.RunsOnce }; // Runs exactly once on startDate.
        }
      } else {
        if (startDate.getTime() === endDate.getTime()) {
          return { jobType: JobType.RunsOnce }; // Runs exactly once on startDate.
        } else {
          return { jobType: JobType.PeriodicWithStartWithEnd }; // Periodic run with start date and end date.
        }
      }
    } else if (options.cronTime && !options.startDate && !options.endDate) { // Runs immediately. Theres is no end date. It runs infinitely
      return { jobType: JobType.PeriodicWoStartWoEnd };
    } else if (options.cronTime && options.startDate && !options.endDate) { // Starts at specfic time. Runs infinitely. Period is defined by crontime
      return { jobType: JobType.PeriodicWithStartWoEnd };
    } else if (options.cronTime && !options.startDate && options.endDate) { // Starts at current time. Runs until a specific time. Period is defined by crontime
      return { jobType: JobType.PeriodicWoStartWithEnd };
    }

    return { jobType: JobType.Invalid, message: 'Options of the TimerTrigger are invalid!' };
  }

  private isValidCrontime(cronTime: string): {result: boolean, message?: string} {
    try {
      new CronTime(cronTime);
      return { result: true };
    } catch (e: any) {
      return {
        result: false,
        message: `Error occured during initializtaion of the cron job! Cron time has a invalid format: ${e.message}.`
      };
    }
  }

  private startCronJob(cronTime: string): void {
    this.cronJob.setTime(new CronTime(cronTime));
    this.cronJob.start();
  }

  private stopCronJob(): void {
    this.cronJob.stop();
    super.reset();
  }

  private scheduleCronJobStart(startDate: Date, cronTime: string): void {
    const startTimeOut = startDate.getTime() - Date.now();
    setTimeout(() => {
      this.startCronJob(cronTime);
    }, startTimeOut);
  }

  private scheduleCronJobStop(endDate: Date): void {
    const endTimeOut = endDate.getTime() - Date.now();
    setTimeout(() => {
      this.stopCronJob();
    }, endTimeOut);
  }

  private scheduleOnce(startDate: Date): void {
    setTimeout(() => {
      super.start();
    }, startDate.getTime() - Date.now());
  }

  private isExpired(startDate: Date, endDate: Date): boolean {
    if ((startDate.getTime() >= Date.now() && endDate.getTime() >= Date.now()) || (endDate.getTime() >= Date.now())) {
      return false;
    }

    return true;
  }
}
