const fs = require('fs');

const checkArguments = (params, args) => {
  const result = {
    success: true,
    errorMsg: '',
    args
  };

  for (let i = 0; i < params.length; i++) {
    if (typeof args[i] === 'undefined') {
      if (params[i].required) {
        result.success = false;
        result.errorMsg = `Missing required parameter: ${params[i].label}`;
        break;
      }
      break;
    }

    if (params[i].type === 'string') {
      if (typeof args[i] !== 'string') {
        try {
          result.args[i] = args[i].toString();
        } catch {
          result.success = false;
          result.errorMsg = `The ${params[i].label} parameter is supposed to be a string`;
          break;
        }
      }
    }

    if (params[i].type === 'boolean') {
      if (typeof args[i] !== 'boolean') {
        if (args[i] === 'true' || args[i] === 1 || args[i] === '1') {
          result.args[i] = true;
        } else if (args[i] === 'false' || args[i] === 0 || args[i] === '0') {
          result.args[i] = false;
        } else {
          result.success = false;
          result.errorMsg = `The ${params[i].label} parameter is supposed to be a boolean`;
          break;
        }
      }
    }

    if (params[i].type === 'float') {
      if (typeof args[i] !== 'number') {
        const convertedFloat = parseFloat(args[i]);
        if (isNaN(convertedFloat)) {
          result.success = false;
          result.errorMsg = `The ${params[i].label} parameter is supposed to be a float`;
          break;
        }
        result.args[i] = convertedFloat;
      }
    }

    if (params[i].type === 'integer') {
      if (typeof args[i] !== 'number') {
        const convertedInt = parseInt(args[i], 10);
        if (isNaN(convertedInt)) {
          result.success = false;
          result.errorMsg = `The ${params[i].label} parameter is supposed to be a integer`;
          break;
        }
        result.args[i] = convertedInt;
      }
    }

    if (params[i].type === 'json') {
      if (typeof args[i] !== 'string') {
        result.success = false;
        result.errorMsg = `The ${params[i].label} parameter is supposed to be a json`;
        break;
      }
      try {
        JSON.parse(args[i]);
      } catch {
        result.success = false;
        result.errorMsg = `The ${params[i].label} parameter is supposed to be a json`;
      }
    }

    if (params[i].type === 'list') {
      if (!Array.isArray(args[i])) {
        result.success = false;
        result.errorMsg = `The ${params[i].label} parameter is supposed to be a list`;
        break;
      }
    }
  }

  return result;
};

const getModuleData = instanceDir => {
  if (!fs.existsSync(instanceDir)) {
    throw new Error(`Instance not exist: ${instanceDir.substr(instanceDir.lastIndexOf('/') + 1)}`);
  }

  const configFilePath = `${instanceDir}/configuration.json`;

  if (!fs.existsSync(configFilePath)) {
    throw new Error('Instance configuration descriptor file not exists!');
  }

  const config = require(configFilePath);

  let modulePath = instanceDir.replace('instances', 'modules');
  modulePath = modulePath.substr(0, modulePath.lastIndexOf('/') + 1) + config.moduleId;

  if (!fs.existsSync(modulePath)) {
    throw new Error('Module directory does not exist!');
  }

  const moduleDescriptorPath = `${modulePath}/moduledescriptor.json`;

  if (!fs.existsSync(moduleDescriptorPath)) {
    throw new Error('Module methods definition file does not exist!');
  }

  let moduleDescriptor = fs.readFileSync(moduleDescriptorPath);
  moduleDescriptor = JSON.parse(moduleDescriptor);

  return moduleDescriptor;
};

const getModulePackageData = instanceDir => {
  const configFilePath = `${instanceDir}/configuration.json`;

  if (!fs.existsSync(configFilePath)) {
    throw new Error('Instance configuration descriptor file not exists!');
  }

  const config = require(configFilePath);

  let modulePath = instanceDir.replace('instances', 'modules');
  modulePath = modulePath.substr(0, modulePath.lastIndexOf('/') + 1) + config.moduleId;

  if (!fs.existsSync(modulePath)) {
    throw new Error('Module directory does not exist!');
  }

  const packageJsonPath = `${modulePath}/package.json`;

  if (!fs.existsSync(packageJsonPath)) {
    throw new Error('Module package definition file does not exist!');
  }

  let packageJson = fs.readFileSync(packageJsonPath);
  packageJson = JSON.parse(packageJson);

  return packageJson;
};

const updateVariable = (name, value, instanceId, comm) => {
    comm.send({
      type: 'updateVariable',
      instanceId: instanceId,
      args: {
        name, 
        value
      }
    })
  }

module.exports = {
  checkArguments,
  getModuleData,
  getModulePackageData,
  updateVariable
};
